% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ode.R
\name{ode}
\alias{ode}
\title{Ordinary Differential Equations}
\usage{
ode(
  f,
  var,
  times,
  timevar = NULL,
  params = list(),
  method = "rk4",
  drop = FALSE
)
}
\arguments{
\item{f}{vector of \code{characters}, or a \code{function} returning a numeric vector, giving the values of the derivatives in the ODE system at time \code{timevar}. See examples.}

\item{var}{vector giving the initial conditions. See examples.}

\item{times}{discretization sequence, the first value represents the initial time.}

\item{timevar}{the time variable used by \code{f}, if any.}

\item{params}{\code{list} of additional parameters passed to \code{f}.}

\item{method}{the solver to use. One of \code{"rk4"} (Runge-Kutta) or \code{"euler"} (Euler).}

\item{drop}{if \code{TRUE}, return only the final solution instead of the whole trajectory.}
}
\value{
Vector of final solutions if \code{drop=TRUE}, otherwise a \code{matrix} with as many 
rows as elements in \code{times} and as many columns as elements in \code{var}.
}
\description{
Solves a numerical or symbolic system of ordinary differential equations.
}
\examples{
## ==================================================
## Example: symbolic system 
## System:  dx = x dt
## Initial: x0 = 1
## ==================================================
f <- "x"
var <- c(x=1)
times <- seq(0, 2*pi, by=0.001)
x <- ode(f, var, times)
plot(times, x, type = "l")

## ==================================================
## Example: time dependent system
## System:  dx = cos(t) dt
## Initial: x0 = 0
## ==================================================
f <- "cos(t)"
var <- c(x=0)
times <- seq(0, 2*pi, by=0.001)
x <- ode(f, var, times, timevar = "t")
plot(times, x, type = "l")

## ==================================================
## Example: multivariate time dependent system
## System:  dx = x dt 
##          dy = x*(1+cos(10*t)) dt
## Initial: x0 = 1
##          y0 = 1
## ==================================================
f <- c("x", "x*(1+cos(10*t))")
var <- c(x=1, y=1)
times <- seq(0, 2*pi, by=0.001)
x <- ode(f, var, times, timevar = "t")
matplot(times, x, type = "l", lty = 1, col = 1:2)

## ==================================================
## Example: numerical system
## System:  dx = x dt 
##          dy = y dt 
## Initial: x0 = 1
##          y0 = 2
## ==================================================
f <- function(x, y) c(x, y)
var <- c(x=1, y=2)
times <- seq(0, 2*pi, by=0.001)
x <- ode(f, var, times)
matplot(times, x, type = "l", lty = 1, col = 1:2)

## ==================================================
## Example: vectorized interface
## System:  dx = x dt 
##          dy = y dt 
##          dz = y*(1+cos(10*t)) dt  
## Initial: x0 = 1
##          y0 = 2
##          z0 = 2
## ==================================================
f <- function(x, t) c(x[1], x[2], x[2]*(1+cos(10*t)))
var <- c(1,2,2)
times <- seq(0, 2*pi, by=0.001)
x <- ode(f, var, times, timevar = "t")
matplot(times, x, type = "l", lty = 1, col = 1:3)

}
\references{
Guidotti E (2022). "calculus: High-Dimensional Numerical and Symbolic Calculus in R." Journal of Statistical Software, 104(5), 1-37. \doi{10.18637/jss.v104.i05}
}
\seealso{
Other integrals: 
\code{\link{integral}()}
}
\concept{integrals}
