/*=========================================================================
 *
 *  Copyright NumFOCUS
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *         https://www.apache.org/licenses/LICENSE-2.0.txt
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *=========================================================================*/

#ifndef itkFEMLoadPoint_h
#define itkFEMLoadPoint_h

#include "itkFEMLoadElementBase.h"
#include "ITKFEMExport.h"
#include "vnl/vnl_vector.h"

namespace itk
{
namespace fem
{
/**
 * \class LoadPoint
 * \brief This load is applied on a point in an element.
 *
 * \ingroup ITKFEM
 */
class ITKFEM_EXPORT LoadPoint : public LoadElement
{
public:
  /** Standard class type aliases. */
  using Self = LoadPoint;
  using Superclass = LoadElement;
  using Pointer = SmartPointer<Self>;
  using ConstPointer = SmartPointer<const Self>;

  /** Method for creation through the object factory. */
  itkSimpleNewMacro(Self);

  /** \see LightObject::GetNameOfClass() */
  itkOverrideGetNameOfClassMacro(LoadPoint);

  /** CreateAnother method will clone the existing instance of this type,
   * including its internal member variables. */
  itk::LightObject::Pointer
  CreateAnother() const override;

  /** Default constructor. */
  LoadPoint()
    : m_Point(2, Float{})
    , m_ForcePoint(2, Float{})
  {
    // Default initialization of 2D point and force vector
  }

  /** Set the point where the load acts. */
  void
  SetPoint(const vnl_vector<Float> p);

  /** Get the point where the load acts. */
  vnl_vector<Float>
  GetPoint();

  /** Set the force vector. */
  void
  SetForce(const vnl_vector<Float> f);

  /** Get the force vector. */
  vnl_vector<Float>
  GetForce();

  /** Apply the load to the specified element.
   * Modified version from the one in itk::fem::LoadLandmark. */
  void
  ApplyLoad(Element::ConstPointer element, Element::VectorType & Fe) override;

protected:
  void
  PrintSelf(std::ostream & os, Indent indent) const override;

  /** Point of which the load acts in global the coordinate system. */
  vnl_vector<Float> m_Point{};

  /** The actual load vector. */
  vnl_vector<Float> m_ForcePoint{};
};
} // end namespace fem
} // end namespace itk

#endif // itkFEMLoadPoint_h
