/*
 * @(#)xdino.c
 *
 * Copyright 1995 - 2025  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Driver file for Dino */

#ifndef WINVER
#include "version.h"
static const char aboutHelp[] = {
"Dino Version " VERSION "\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd AT verizon.net>\n"
"The latest version is at: "
"https://www.sillycycle.com/puzzles.html"
};

static const char synopsisHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-{border|bd} {color}] [-face{0|1|2|3|4|5} {color}]\n"
"[-frame {color}] [-delay msecs] [-[no]sound]\n"
"[-moveSound {filename}] [-{font|fn} {fontname}]\n"
"[-view {int}] [-{mode {int} | both}] [-[no]orient]\n"
"[-[no]practice] [-userName {string}]\n"
"[-scoreFile {filename}] [-scores] [-version]"
};
#endif

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA) || defined(WINVER)
static const char descriptionHelp[] = {
"The period 2 turning (i.e. the edges turn with 180 "
"degree intervals) was suggested by Derek Bosch\n"
"<bosch@sgi.com>.  The physical puzzle has period 3 "
"turning (i.e. the points turn with 120 degree\n"
"intervals) is called a Triangle - 4 Cube - Dinosaur "
"with colored dinosaurs on it, (or Triangle - 4 Cube -\n"
"6 Colors with solid colors, (a 4 color and a 2 color "
"are also available with point oriented coloring))\n"
"manufactured by The International Puzzle & Games "
"(Great Animation International Corporation)."
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a piece.  Release "
"\"mouse-left\" button on a piece on the same face.\n"
"The pieces will then turn towards where the mouse "
"button was released.\n"
"\n"
"Click \"mouse-center\" button, or press \"P\" or \"p\" "
"keys, to toggle the practice mode (in practice mode the\n"
"record should say \"practice\").  This is good for learning "
"moves and experimenting.\n"
"\n"
"Click \"mouse-right\" button, or press \"Z\" or \"z\" "
"keys, to randomize the puzzle (this must be done first\n"
"to set a new record).\n"
"\n"
"Press \"G\" or \"g\" keys to read (get) a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"C\" or \"c\" keys to clear the puzzle.\n"
"\n"
"Press \"S\" or \"s\" keys to start auto-solver.\n"
"\n"
"Press \"O\" or \"o\" keys to toggle the orient mode.  One "
"has to orient the faces in orient mode, besides\n"
"getting all the faces to be the same color.  To do this "
"one has to get the lines to be oriented in\n"
"the same direction.  This does add complexity (ever so "
"slightly, there are only 2 possibilities) so there\n"
"are 2 sets of records.\n"
"\n"
"Press \"2\", \"3\", \"B\", or \"b\" keys (not the keypad "
"2, 3) to change modes to Period 2, Period 3, or Both.\n"
"\n"
"Press \"V\" or \"v\" keys to change the view of the puzzle.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of pieces.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of pieces.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Dino2d as:\n"
"7 8 9  Upper Left, Up, Upper Right\n"
"  ^\n"
"4< >6  Left, Right\n"
"  v\n"
"1 2 3  Lower Left, Down, Lower Right\n"
"Note: Up, Left, Right, and Down only work when the "
"control key is pressed and there is no analog for\n"
"Dino3d.\n"
"\n"
"Key pad for Dino3d, use must use your intuition (is this "
"a cop out or what?).  The key pad is defined\n"
"differently depending on which side of the cube your mouse "
"is pointing at.\n"
"\n"
"Use the alt key and the left mouse button, keypad, or "
"arrow keys to move the center of the cube.  The 2\n"
"opposite corners do not move.\n"
"\n"
"Use the shift keys and the left mouse button, keypad, "
"or arrow key to access \"Period 2\" turns from \"Both\"\n"
"mode, otherwise it assumes \"Period 3\" turning.  Edges "
"turn in \"Period 2\" and corners turn in \"Period 3\".\n"
"The \"Period 2\" mode has extra cuts around the faces.\n"
"\n"
"Use the control key and the left mouse button, keypad, or "
"arrow keys to move the whole cube.  This is\n"
"not recorded as a turn."
};

static const char referencesHelp[] = {
"International Puzzles & Games Catalog."
};
#endif

#include "file.h"
#ifdef WINVER
#include "DinoP.h"
#define TITLE "wdino"

static DinoRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "C:/ProgramData/wpuzzles"
#else
#define SCOREPATH "C:\\ProgramData\\wpuzzles"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->dino.started = b
#else
#include "xwin.h"
#include "xgui.h"
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Dino.h"
#include "Dino2d.h"
#ifdef HAVE_OPENGL
#include "DinoGL.h"
#else
#include "Dino3d.h"
#endif
#ifdef HAVE_XPM
#include <X11/xpm.h>
#ifdef CONSTPIXMAPS
#include "icons/16x16/dino.xpm"
#include "icons/22x22/dino.xpm"
#include "icons/24x24/dino.xpm"
#include "icons/32x32/dino.xpm"
#include "icons/48x48/dino.xpm"
#include "icons/64x64/dino.xpm"
#else
#include "pixmaps/16x16/dino.xpm"
#include "pixmaps/22x22/dino.xpm"
#include "pixmaps/24x24/dino.xpm"
#include "pixmaps/32x32/dino.xpm"
#include "pixmaps/48x48/dino.xpm"
#include "pixmaps/64x64/dino.xpm"
#endif
#define RESIZE_XPM(s) ((char **) (((s)<=32)?\
(((s)<=22)?(((s)<=16)?dino_16x16:dino_22x22):\
(((s)<=24)?dino_24x24:dino_32x32)):\
(((s)<=48)?dino_48x48:dino_64x64)))
#endif
#include "pixmaps/64x64/dino.xbm"
#define DEFINE_XBM (char *) dino_64x64_bits, dino_64x64_width, dino_64x64_height
#ifdef HAVE_EDITRES
#ifdef __VMS
#include <Xmu/Editres.h>
#else
#include <X11/Xmu/Editres.h>
#endif
#endif
#ifndef SCOREPATH
#ifdef __VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef SCOREFILE
#define SCOREFILE "dino.scores"
#endif

#define NEVER (-1)
#define USER_NAME_LENGTH 120
#define MESSAGE_LENGTH (USER_NAME_LENGTH+64)
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USER_NAME_LENGTH];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[MAX_MODES][2];
static int movesDsp = 0;
static char messageDsp[MESSAGE_LENGTH] = "Welcome";
static char recordDsp[MESSAGE_LENGTH] = "NOT RECORDED";

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static Widget movesText, recordText, message;
static Widget soundMenuItem;
static Widget orientToggle, practiceToggle;
static Widget practiceDialog, randomizeDialog;
static Widget descriptionDialog, featuresDialog;
static Widget synopsisDialog, referencesDialog, aboutDialog;
static char buff[MESSAGE_LENGTH];
static const char *modeLabel = "Period:";
static const char *speedLabel = "Animation Speed:";
#define PRINT_MESSAGE(b) printState(message, b)
#define MIN_SPEED 1
#define MAX_SPEED 50
static const char *modeStrings[] =
{
	"Period 2", "Period 3", "Both"
};
#elif !defined(WINVER)
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#ifdef HAVE_MOTIF
static Widget modeRadio[MAX_MODES];
static Widget speedChanger;
#elif defined(HAVE_ATHENA)
#ifdef USE_POPUP
static Widget modeNameLabel;
#else
static Widget modeRadio[MAX_MODES];
#endif
static Widget speedSlider, speedSliderLabel;
static const char *fileTypes[] =
{
	"Get",
	"Write",
	"Exit"
};
#define numFileTypes (sizeof(fileTypes)/sizeof(fileTypes[0]))
static const char *playTypes[] =
{
	"Undo",
	"Redo",
	"Clear",
#ifdef EXTRA
	"Practice",
#endif
	"Randomize",
	"Auto-solve",
#ifdef EXTRA
	"Oriented",
	"Speed >",
	"Slow <",
#endif
	"Sound @"
#ifdef HAVE_OPENGL
	, "View"
#endif
};
#define numPlayTypes (sizeof(playTypes)/sizeof(playTypes[0]))
static const char *helpTypes[] =
{
	"Description...",
	"Features...",
	"Synopsis...",
	"References...",
	"About..."
};
#define numHelpTypes (sizeof(helpTypes)/sizeof(helpTypes[0]))
#else
#ifndef WINVER
static Widget shell;
#endif
#endif
static char scoreFileName[FILE_NAME_LENGTH] = SCOREFILE;
static char fileName[FILE_NAME_LENGTH];
static Boolean randomized = False;
#ifdef WINVER
#define PROGRAM_NAME_LENGTH 80
static char progDsp[PROGRAM_NAME_LENGTH] = TITLE;
static char userNameDsp[USER_NAME_LENGTH] = "guest";
#else
static Pixmap pixmap = None;
static Widget topLevel, puzzle2d, puzzle3d;
static char userNameDsp[USER_NAME_LENGTH] = "";
#define WINDOW_HEIGHT 300

#ifdef HAVE_MOTIF
static void
printState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("printState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w,
		XmNlabelString, xmstr, NULL);
	XmStringFree(xmstr);
}
#elif defined(HAVE_ATHENA)
static void
printState(Widget w, char *msg)
{
	XtVaSetValues(w,
		XtNlabel, msg, NULL);
	if (w == movesText)
		XtVaSetValues(w,
			XtNwidth, 64, NULL);
}
#endif

static void
printRecords(void)
{
	int mode, orient;

	(void) printf("      DINO  HALL OF FAME\n\n");
	(void) printf("MODE ORIENT USER            MOVES\n");
	for (mode = 0; mode < MAX_MODES; mode++)
		for (orient = 0; orient < 2; orient++) {
			if (puzzleRecord[mode][orient].score > 0)
				(void) printf("%4d%7d %-12s%9d\n",
					mode + PERIOD2, orient,
					puzzleRecord[mode][orient].name,
					puzzleRecord[mode][orient].score);
		}
}
#endif

static void
initRecords(void)
{
	int mode, orient;

	for (mode = 0; mode < MAX_MODES; mode++)
		for (orient = 0; orient < 2; orient++) {
			puzzleRecord[mode][orient].score = NEVER;
			(void) strncpy(puzzleRecord[mode][orient].name, NOACCESS,
				USER_NAME_LENGTH);
		}
}

static void
readRecords(void)
{
	FILE *fp;
	int n, mode, orient;
	char userName[USER_NAME_LENGTH];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USER_NAME_LENGTH);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USER_NAME_LENGTH);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Cannot read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Cannot read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (mode = 0; mode < MAX_MODES; mode++)
		for (orient = 0; orient < 2; orient++) {
			if (fscanf(fp, "%d %s\n", &n, userName) != 2) {
				(void) fprintf(stderr,
					"corrupt record, expecting integer and name for %d %d\n",
					mode, orient);
				(void) fclose(fp);
				return;
			}
			if (n <= puzzleRecord[mode][orient].score ||
					puzzleRecord[mode][orient].score <= NEVER) {
				puzzleRecord[mode][orient].score = n;
				(void) strncpy(puzzleRecord[mode][orient].name,
					userName, USER_NAME_LENGTH);
			}
		}
	(void) fclose(fp);
}

static void
writeRecords(void)
{
	FILE *fp;
	int mode, orient;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Cannot write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILE_NAME_LENGTH];

		(void) strncpy(lockFile, fileName, FILE_NAME_LENGTH - 6);
		lockFile[FILE_NAME_LENGTH - 6] = '\0';
		(void) strncat(lockFile, ".lock", 6);
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (mode = 0; mode < MAX_MODES; mode++) {
			for (orient = 0; orient < 2; orient++)
				(void) fprintf(fp, "%d %s\n",
					puzzleRecord[mode][orient].score,
					puzzleRecord[mode][orient].name);
			if (mode < MAX_MODES - 1)
				(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
printRecord(int mode, Boolean orient, Boolean practice)
{
	int i = mode - PERIOD2;
	int j = (orient) ? 1 : 0;

	if (practice) {
		(void) strncpy(recordDsp, "practice", 9);
	} else if (puzzleRecord[i][j].score <= NEVER) {
		(void) strncpy(recordDsp, "NEVER ", 7);
		(void) strncat(recordDsp, NOACCESS, strlen(NOACCESS) + 1);
	} else {
#ifdef HAVE_SNPRINTF
		(void) snprintf(recordDsp, MESSAGE_LENGTH,"%d %s",
			puzzleRecord[i][j].score,
			puzzleRecord[i][j].name);
#else
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[i][j].score,
			puzzleRecord[i][j].name);
#endif
	}
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printState(recordText, recordDsp);
#endif
}

#ifndef WINVER
/* There is probably a better way to assure that they are the same
 * but I do not know it off hand. */
static void
makeEquivalent(String *userName, String *scoreFile, int *mode,
		Boolean *orient, Boolean *practice, int *delay)
{
	Boolean mono, reverse;
	Boolean scoreOnly, versionOnly;
	Pixel foreground, background, pieceBorder;
	String faceColor[MAX_FACES];

	XtVaGetValues(puzzle2d,
		XtNuserName, userName,
		XtNscoreFile, scoreFile,
		XtNmode, mode,
		XtNorient, orient,
		XtNpractice, practice,
		XtNmono, &mono,
		XtNreverseVideo, &reverse,
		XtNdelay, delay,
		XtNforeground, &foreground,
		XtNbackground, &background,
		XtNpieceBorder, &pieceBorder,
		XtNfaceColor0, &(faceColor[0]),
		XtNfaceColor1, &(faceColor[1]),
		XtNfaceColor2, &(faceColor[2]),
		XtNfaceColor3, &(faceColor[3]),
		XtNfaceColor4, &(faceColor[4]),
		XtNfaceColor5, &(faceColor[5]),
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf("%s\n", aboutHelp);
		exit(0);
	}
	if (strcmp(*scoreFile, "")) {
		(void) strncpy(scoreFileName, *scoreFile, FILE_NAME_LENGTH - 1);
		scoreFileName[FILE_NAME_LENGTH - 1] = '\0';
	}
	if (scoreOnly) {
		initRecords();
		readRecords();
		printRecords();
		exit(0);
	}
	XtVaSetValues(puzzle2d,
		XtNdirection, IGNORE_DIR,
		XtNstart, False, NULL);
	XtVaSetValues(puzzle3d,
		XtNuserName, *userName,
		XtNmode, *mode,
		XtNorient, *orient,
		XtNpractice, *practice,
		XtNmono, mono,
		XtNreverseVideo, reverse,
		XtNdelay, *delay,
		XtNdirection, IGNORE_DIR,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
}
#endif

static void
printStatus(char *msg, int nMoves
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		, int dim, int mode
#endif
		)
{
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printState(message, msg);
	intCpy(buff, nMoves);
	printState(movesText, buff);
#else
	char titleDsp[TITLE_LENGTH] = "";
	char dimDsp[3];
	char mb[10];
#ifdef HAVE_OPENGL
	if (dim == 4)
		(void) strncpy(dimDsp, "GL", 3);
	else
#endif
	if (dim == 3)
		(void) strncpy(dimDsp, "3d", 3);
	else
		(void) strncpy(dimDsp, "2d", 3);
	if (mode == BOTH)
		(void) strncpy(mb, "both", 5);
	else
		intCpy(mb, mode);
#ifdef HAVE_SNPRINTF
	(void) snprintf(titleDsp, TITLE_LENGTH,
		"%s%s.%s: (%d/%s) - %s",
		progDsp, dimDsp, mb, nMoves, recordDsp, msg);
#else
	(void) sprintf(titleDsp,
		"%s%s.%s: (%d/%s) - %s",
		progDsp, dimDsp, mb, nMoves, recordDsp, msg);
#endif
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent((dim == 2) ? puzzle2d : puzzle3d),
		XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
handleSolved(int counter, int mode, Boolean orient)
{
	int i = mode - PERIOD2;
	int j = (orient) ? 1 : 0;

	if (counter < puzzleRecord[i][j].score ||
			puzzleRecord[i][j].score <= NEVER) {
		readRecords();	/* Maybe its been updated by another */
		puzzleRecord[i][j].score = counter;
		(void) strncpy(puzzleRecord[i][j].name, userNameDsp,
			USER_NAME_LENGTH);
		if (orient && (counter < puzzleRecord[!i][j].score ||
				puzzleRecord[!i][j].score <= NEVER)) {
			puzzleRecord[!i][j].score = counter;
			(void) strncpy(puzzleRecord[!i][j].name, userNameDsp,
				USER_NAME_LENGTH);
		}
		writeRecords();
		printRecord(mode, orient, False);
		return True;
	}
	return False;
}

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static int
getSpeed(int delay)
{
	return MAX_SPEED + MIN_SPEED - delay - 1;
}

static int
getDelay(int speed)
{
	return MAX_SPEED + MIN_SPEED - speed - 1;
}

static void
orientCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
	int mode;
	Boolean orient, practice, old;

#ifdef HAVE_MOTIF
	orient = cbs->set;
#elif defined(HAVE_ATHENA)
	orient = state;
#endif
	XtVaGetValues(puzzle2d,
		XtNmode, &mode,
		XtNorient, &old,
		XtNpractice, &practice, NULL);
	if (old == orient)
		return;
	XtVaSetValues(puzzle3d,
		XtNorient, orient, NULL);
	XtVaSetValues(puzzle2d,
		XtNorient, orient, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(mode, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}

static void
practiceCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
	int mode;
	Boolean orient, practice, old;

#ifdef HAVE_MOTIF
	practice = cbs->set;
#elif defined(HAVE_ATHENA)
	practice = state;
#endif
	XtVaGetValues(puzzle2d,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNpractice, &old, NULL);
	if (old == practice)
		return;
	XtVaSetValues(puzzle3d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaSetValues(puzzle2d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(mode, orient, practice);
	if (practice)
		messageDsp[0] = '\0';
	else
		(void) strncpy(messageDsp, "Randomize (z) to start",
			MESSAGE_LENGTH);
	printState(message, messageDsp);
}

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
speedUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int delay, speed;

	XtVaGetValues(puzzle2d,
		XtNdelay, &delay, NULL);
	speed = getSpeed(delay);
	speed++;
	if (speed > MAX_SPEED) {
		return;
	}
	setScale(speedSlider, speedSliderLabel, speed,
		MIN_SPEED, MAX_SPEED, True);
	delay = getDelay(speed);
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}

static void
speedDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int delay, speed;

	XtVaGetValues(puzzle2d,
		XtNdelay, &delay, NULL);
	speed = getSpeed(delay);
	speed--;
	if (speed < MIN_SPEED) {
		return;
	}
	setScale(speedSlider, speedSliderLabel, speed,
		MIN_SPEED, MAX_SPEED, True);
	delay = getDelay(speed);
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}

#else

static void
speedChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int delay, speed, old;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	speed = cbs->position;
#else
	speed = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &speed,
			MIN_SPEED, MAX_SPEED, SCROLL_SIZE)) {
		return;
	}
#endif
	delay = getDelay(speed);
	XtVaGetValues(puzzle2d,
		XtNdelay, &old, NULL);
	if (old == delay)
		return;
#ifdef HAVE_ATHENA
	setScale(w, label, speed, MIN_SPEED, MAX_SPEED, True);
#endif
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}
#endif

static void
modeCallback(Widget w,
#ifdef HAVE_MOTIF
		int position, XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer clientData, XtPointer callData
#endif
		)
{
	Boolean orient, practice, state = True;
	int mode, old;
#ifdef HAVE_MOTIF
	if (!cbs->set)
		return;
#elif defined(HAVE_ATHENA)
	int position = (size_t) clientData;
#endif

	XtVaGetValues(puzzle2d,
		XtNmode, &old,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	mode = position + PERIOD2;
	if (old == mode)
		return;
#ifdef HAVE_ATHENA
#ifdef USE_POPUP
	XtVaSetValues(modeNameLabel,
		XtNlabel, modeStrings[position], NULL);
#else
	XtVaGetValues(modeRadio[position],
		XtNstate, &state, NULL);
#endif
#endif
	if (state) {
		XtVaSetValues(puzzle3d,
			XtNmode, mode, NULL);
		XtVaSetValues(puzzle2d,
			XtNmode, mode, NULL);
	}
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(mode, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}

static void
puzzlePracticeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmAnyCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
#ifdef HAVE_MOTIF
	if (cbs->reason == XmCR_OK)
#elif defined(HAVE_ATHENA)
	XtPopdown(practiceDialog);
#endif
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_PRACTICE, NULL);
}

static void
puzzleRandomizeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmAnyCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
#ifdef HAVE_MOTIF
	if (cbs->reason == XmCR_OK)
#elif defined(HAVE_ATHENA)
	XtPopdown(randomizeDialog);
#endif
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_RANDOMIZE, NULL);
}

static void
fileMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData) + ACTION_GET;

	if (value == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle2d,
		XtNmenu, value, NULL);
}

static void
setSoundCheck(Widget w)
{
	Boolean sound;
	XtVaGetValues(puzzle2d,
		XtNsound, &sound, NULL);
#ifdef HAVE_MOTIF
	setToggle(w, sound, False);
#elif defined(HAVE_ATHENA)
	menuCheck(topLevel, w, sound);
#endif
}

static void
playMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData) + ACTION_UNDO;

#ifndef EXTRA
	if (value >= ACTION_PRACTICE)
		value += ACTION_RANDOMIZE - ACTION_PRACTICE;
	if (value >= ACTION_ORIENTIZE)
		value += ACTION_SOUND - ACTION_ORIENTIZE;
#endif
	XtVaSetValues(puzzle2d,
		XtNmenu, value, NULL);
	if (value == ACTION_SOUND)
		setSoundCheck(w);
}

static void
helpMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	Widget dialog;

#ifdef HAVE_ATHENA
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
#endif
	switch (value) {
	case 0:
		dialog = descriptionDialog;
		break;
	case 1:
		dialog = featuresDialog;
		break;
	case 2:
		dialog = synopsisDialog;
		break;
	case 3:
		dialog = referencesDialog;
		break;
	case 4:
		dialog = aboutDialog;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpMenuCallback: %d", value);
			XtWarning(buf);
			free(buf);
			return;
		}
	}
#ifdef HAVE_MOTIF
	XtManageChild(dialog);
#elif defined(HAVE_ATHENA)
	XtPopup(dialog, XtGrabNone);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(dialog), &wmDeleteWindow, 1);
#endif
}
#endif

static void
initialize(
#ifdef WINVER
DinoWidget w, HBRUSH brush
#else
void
#endif
)
{
	int mode;
	Boolean orient, practice;
	char *userName, *scoreFile;
#ifdef WINVER
	int dim;

	initializePuzzle(w, brush);

	mode = w->dino.mode;
	orient = w->dino.orient;
	practice = w->dino.practice;
	userName = w->dino.userName;
	scoreFile = w->dino.scoreFile;
	dim = w->dino.dim;
	SET_STARTED(w, False);
	if (strcmp(scoreFile, "")) {
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH - 1);
		scoreFileName[FILE_NAME_LENGTH - 1] = '\0';
	}
#else
	int delay;
	makeEquivalent(&userName, &scoreFile, &mode,
		&orient, &practice, &delay);
#ifdef HAVE_MOTIF
	setToggle(modeRadio[mode - PERIOD2], True, True);
	setToggle(orientToggle, orient, False);
	setToggle(practiceToggle, practice, False);
	{
		int speed;
		speed = getSpeed(delay);
#ifdef USE_SPIN
		XtVaSetValues(speedChanger,
			XmNposition, speed, NULL);
#else
		XmScaleSetValue(speedChanger, speed);
#endif
	}
	setSoundCheck(soundMenuItem);
#elif defined(HAVE_ATHENA)
	XtVaSetValues(orientToggle,
		XtNstate, orient, NULL);
	XtVaSetValues(practiceToggle,
		XtNstate, practice, NULL);
	{
		int speed;
		speed = getSpeed(delay);
		setScale(speedSlider, speedSliderLabel, speed,
			MIN_SPEED, MAX_SPEED, True);
	}
	setSoundCheck(soundMenuItem);
#endif
#endif
	initRecords();
	readRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH - 1);
	userNameDsp[USER_NAME_LENGTH - 1] = '\0';
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH - 1);
		userNameDsp[USER_NAME_LENGTH - 1] = '\0';
#else
		char *login;

#ifdef HAVE_CUSERID
		login = cuserid(NULL);
#else
		login = getlogin();
#endif
		if (login == NULL) {
			userNameDsp[0] = '\0';
		} else {
			(void) strncpy(userNameDsp, login, USER_NAME_LENGTH - 1);
			userNameDsp[USER_NAME_LENGTH - 1] = '\0';
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) strncpy(userNameDsp, "guest", 6);
#endif
	}
	printRecord(mode, orient, practice);
#ifdef WINVER
	printStatus(messageDsp, movesDsp, dim, mode);
#else
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printStatus(messageDsp, movesDsp);
#else
	printStatus(messageDsp, movesDsp, 2, mode);
#ifdef HAVE_OPENGL
	printStatus(messageDsp, movesDsp, 4, mode);
#else
	printStatus(messageDsp, movesDsp, 3, mode);
#endif
#endif
#endif
}

#ifdef WINVER
void
setPuzzle(DinoWidget w, int reason)
#else
static void
puzzleCallback(Widget w, caddr_t clientData,
		dinoCallbackStruct *callData)
#endif
{
	int mode;
	Boolean orient, practice, cheat;
#ifdef WINVER
	int dim = 0;
#else
	Boolean start;
	int reason = callData->reason;
	Widget otherw = (Widget) NULL;
#if defined(HAVE_ATHENA) && defined(USE_POPUP)
	int max;
	char *defaultString;
#endif
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
	int dim = 0, otherdim = 0;
#endif

	if (w == puzzle2d) {
		otherw = puzzle3d;
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		dim = 2;
#ifdef HAVE_OPENGL
		otherdim = 4;
#else
		otherdim = 3;
#endif
#endif
	} else if (w == puzzle3d) {
		otherw = puzzle2d;
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		otherdim = 2;
#ifdef HAVE_OPENGL
		dim = 4;
#else
		dim = 3;
#endif
#endif
	}
#endif
	messageDsp[0] = '\0';
#ifdef WINVER
	mode = w->dino.mode;
	orient = w->dino.orient;
	practice = w->dino.practice;
	cheat = w->dino.cheat;
	/*start = w->dino.started;*/
	dim = w->dino.dim;
#else
	XtVaGetValues(w,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		(void) XIconifyWindow(XtDisplay(shell),
			XtWindow(shell),
			XScreenNumberOfScreen(XtScreen(shell)));
#endif
#endif
		break;
#ifndef WINVER
	case ACTION_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(practiceDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(practiceDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(practiceDialog), &wmDeleteWindow, 1);
#else
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_PRACTICE, NULL);
#endif
		break;
	case ACTION_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(randomizeDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(randomizeDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(randomizeDialog), &wmDeleteWindow, 1);
#else
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_RANDOMIZE, NULL);
#endif
		break;
#endif
	case ACTION_RESTORE:
		if (practice) {
			(void) strncpy(recordDsp, "practice",
				MESSAGE_LENGTH);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
			printState(recordText, recordDsp);
#endif
		}
		movesDsp = 0;
		randomized = False;
#ifndef WINVER
		XtVaSetValues(otherw,
			XtNdirection, RESTORE_DIR, NULL);
		XtVaSetValues(w,
			XtNdirection, RESTORE_DIR, NULL);
#endif
		break;
	case ACTION_CLEAR:
		movesDsp = 0;
		randomized = False;
#ifndef WINVER
		XtVaSetValues(otherw,
			XtNdirection, CLEAR_DIR, NULL);
		XtVaSetValues(w,
			XtNdirection, CLEAR_DIR, NULL);
#endif
		break;
	case ACTION_RESET:
		movesDsp = 0;
		randomized = False;
		break;
	case ACTION_AMBIGUOUS:
		(void) strncpy(messageDsp, "Ambiguous move",
			MESSAGE_LENGTH);
		break;
	case ACTION_ILLEGAL:
		if (practice || randomized)
			(void) strncpy(messageDsp, "Illegal move",
				MESSAGE_LENGTH);
		else
			(void) strncpy(messageDsp,
				"Randomize (z) to start", MESSAGE_LENGTH);
		break;
#ifndef WINVER
	case ACTION_MOVED:
		if (!callData->control)
			movesDsp++;
#ifdef DEBUG
		(void) printf("MOVED: face %d, position %d, direction %d, style %d, control %d\n",
			callData->face, callData->position, callData->direction,
			callData->style, callData->control);
#endif
		XtVaSetValues(otherw,
			XtNface, callData->face,
			XtNpos, callData->position,
			XtNdirection, callData->direction,
			XtNstyle, callData->style,
			XtNcontrol, callData->control,
			XtNfast, callData->fast,
			XtNstart, True, NULL);
		SET_STARTED(w, True);
		break;
#endif
	case ACTION_SOLVED:
		if (practice) {
			movesDsp = 0;
		} else if (cheat) {
			(void) strncpy(messageDsp, "No cheating ", 13);
			(void) strncat(messageDsp, userNameDsp, USER_NAME_LENGTH);
			(void) strncat(messageDsp, "!!", 3);
		} else if (handleSolved(movesDsp, mode, orient)) {
			(void) strncpy(messageDsp, "Congratulations ", 17);
			(void) strncat(messageDsp, userNameDsp, USER_NAME_LENGTH);
			(void) strncat(messageDsp, "!!", 3);
		} else {
			(void) strncpy(messageDsp, "Solved!", 8);
		}
		SET_STARTED(w, False);
#ifndef WINVER
		SET_STARTED(otherw, False);
#endif
		randomized = False;
		break;
	case ACTION_COMPUTED:
		SET_STARTED(w, False);
#ifndef WINVER
		SET_STARTED(otherw, False);
#endif
		break;
	case ACTION_PRACTICE:
		movesDsp = 0;
		randomized = False;
		practice = !practice;
		if (!practice)
			(void) strncpy(messageDsp, "Randomize (z) to start",
				MESSAGE_LENGTH);
		printRecord(mode, orient, practice);
#ifdef WINVER
		w->dino.practice = practice;
		w->dino.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, practice,
			XtNstart, False, NULL);
		XtVaSetValues(otherw,
			XtNpractice, practice,
			XtNstart, False, NULL);
#ifdef HAVE_MOTIF
		setToggle(practiceToggle, practice, False);
#elif defined(HAVE_ATHENA)
		XtVaSetValues(practiceToggle,
			XtNstate, practice, NULL);
#endif
#endif
		break;
	case ACTION_RANDOMIZE:
		movesDsp = 0;
		randomized = True;
#ifdef WINVER
		w->dino.practice = False;
		w->dino.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, False,
			XtNstart, False, NULL);
		XtVaSetValues(otherw,
			XtNpractice, False,
			XtNstart, False, NULL);
#endif
		break;
#ifdef HAVE_OPENGL
	case ACTION_VIEW:
		{
			int view;

#ifdef WINVER
			view = w->dino.view;
#else
			XtVaGetValues(puzzle3d,
				XtNview, &view, NULL);
#endif
			view = (view + 1) % (2 * MAX_VIEWS);
#ifdef WINVER
			w->dino.view = view;
#else
			XtVaSetValues(puzzle3d,
				XtNview, view, NULL);
#endif
		}
		break;
#endif
	case ACTION_ORIENTIZE:
		movesDsp = 0;
		orient = !orient;
		printRecord(mode, orient, practice);
#ifdef WINVER
		w->dino.orient = orient;
#else
		XtVaSetValues(w,
			XtNorient, orient, NULL);
		XtVaSetValues(otherw,
			XtNorient, orient, NULL);
#ifdef HAVE_MOTIF
		setToggle(orientToggle, orient, False);
#elif defined(HAVE_ATHENA)
		XtVaSetValues(orientToggle,
			XtNstate, orient, NULL);
#endif
#endif
		break;
	case ACTION_PERIOD2:
	case ACTION_PERIOD3:
	case ACTION_BOTH:
		movesDsp = 0;
		mode = reason - ACTION_PERIOD2 + PERIOD2;
		printRecord(mode, orient, practice);
#ifdef WINVER
		w->dino.mode = mode;
#else
		XtVaSetValues(w,
			XtNmode, mode, NULL);
		XtVaSetValues(otherw,
			XtNmode, mode, NULL);
#ifdef HAVE_MOTIF
		setToggle(modeRadio[mode - PERIOD2], True, True);
#elif defined(HAVE_ATHENA)
#ifdef USE_POPUP
		max = findMaxLength((char **) modeStrings,
			sizeof(modeStrings) / sizeof(*modeStrings));
		createBlank(&defaultString, max, (char *) modeStrings[mode - PERIOD2], 0);
		XtVaSetValues(modeNameLabel,
			XtNlabel, defaultString, NULL);
		free(defaultString);
#else
		XtVaSetValues(modeRadio[mode - PERIOD2],
			XtNstate, True, NULL);
#endif
#endif
#endif
		break;
#ifndef WINVER
	case ACTION_UNDO:
		movesDsp--;
		XtVaSetValues(otherw,
			XtNstart, True,
			XtNface, callData->face,
			XtNpos, callData->position,
			XtNdirection, callData->direction,
			XtNstyle, callData->style, NULL);
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		XtVaSetValues(otherw,
			XtNstart, True,
			XtNface, callData->face,
			XtNpos, callData->position,
			XtNdirection, callData->direction,
			XtNstyle, callData->style, NULL);
		SET_STARTED(w, True);
		break;
#endif
#ifdef WINVER
	case ACTION_DIM:
		dim++;
#ifdef HAVE_OPENGL
		if (dim == 3)
			dim = 4;
		else if (dim > 4)
#else
		if (dim > 3)
#endif
			dim = 2;
		w->dino.dim = dim;
		break;
#endif
#ifndef WINVER
	case ACTION_CHEAT:
		XtVaSetValues(w,
			XtNcheat, cheat, NULL);
		XtVaSetValues(otherw,
			XtNcheat, cheat, NULL);
		break;
#endif
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	case ACTION_SOUND:
		setSoundCheck(soundMenuItem);
		break;
#endif
	}
#ifdef WINVER
	printStatus(messageDsp, movesDsp, dim, mode);
#else
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printStatus(messageDsp, movesDsp);
#else
	printStatus(messageDsp, movesDsp, dim, mode);
	printStatus(messageDsp, movesDsp, otherdim, mode);
#endif
#endif
}

#ifdef WINVER
void
setPuzzleMove(DinoWidget w, int reason, int face, int position, int direction,
		int style, int control, int fast)
{
	int mode;
	/* Boolean orient, practice, cheat; */
	int dim;

	messageDsp[0] = '\0';
	mode = w->dino.mode;
#if 0
	orient = w->dino.orient;
	practice = w->dino.practice;
	cheat = w->dino.cheat;
#endif
	dim = w->dino.dim;
	switch (reason) {
	case ACTION_MOVED:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	case ACTION_CONTROL:
		break;
	case ACTION_UNDO:
		movesDsp--;
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	}
	printStatus(messageDsp, movesDsp, dim, mode);
}

static LRESULT CALLBACK
about(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND && LOWORD(wParam) == IDOK) {
		(void) EndDialog(hDlg, TRUE);
		return TRUE;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	int shift = 0;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.dino.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			enterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.dino.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			leavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		initialize(&widget, brush);
		break;
	case WM_DESTROY:
		destroyPuzzle(brush);
		break;
	case WM_SIZE:
		resizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		exposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		randomizePuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		selectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0,
			(GetKeyState(VK_TAB) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		releasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			((GetKeyState(VK_SHIFT) >> 1) ||
			(GetKeyState(VK_CAPITAL) & 1)),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0,
			(GetKeyState(VK_TAB) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					TOP,
					((GetKeyState(VK_SHIFT) >> 1) ||
					(GetKeyState(VK_CAPITAL) & 1)),
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0,
					(GetKeyState(VK_TAB) >> 1) ? 1 : 0);
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					BOTTOM,
					((GetKeyState(VK_SHIFT) >> 1) ||
					(GetKeyState(VK_CAPITAL) & 1)),
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0,
					(GetKeyState(VK_TAB) >> 1) ? 1 : 0);
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			getPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			writePuzzle(&widget);
			break;
		case ACTION_EXIT:
			destroyPuzzle(brush);
			break;
		case ACTION_HIDE:
			hidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			undoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			redoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			clearPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_PRACTICE:
			practicePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_RANDOMIZE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			randomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			solvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_ORIENTIZE:
			orientizePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DIM:
			(void) dimPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_VIEW:
			(void) viewPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SHIFT_TR:
		case ACTION_SHIFT_BR:
		case ACTION_SHIFT_BL:
		case ACTION_SHIFT_TL:
		case ACTION_SHIFT_CW:
		case ACTION_SHIFT_CCW:
		case ACTION_SHIFT_TOP:
		case ACTION_SHIFT_RIGHT:
		case ACTION_SHIFT_BOTTOM:
		case ACTION_SHIFT_LEFT:
		case ACTION_TR:
		case ACTION_BR:
		case ACTION_BL:
		case ACTION_TL:
		case ACTION_CW:
		case ACTION_CCW:
		case ACTION_TOP:
		case ACTION_RIGHT:
		case ACTION_BOTTOM:
		case ACTION_LEFT:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				shift = (ACTION_SHIFT(LOWORD(wParam))) ? 1 : 0;
				shift = shift || (GetKeyState(VK_CAPITAL) & 1);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) -
					((shift) ? ACTION_SHIFT_TR : ACTION_TR),
					shift, FALSE, FALSE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_SHIFT_CONTROL_TR:
		case ACTION_SHIFT_CONTROL_BR:
		case ACTION_SHIFT_CONTROL_BL:
		case ACTION_SHIFT_CONTROL_TL:
		case ACTION_SHIFT_CONTROL_CW:
		case ACTION_SHIFT_CONTROL_CCW:
		case ACTION_SHIFT_CONTROL_TOP:
		case ACTION_SHIFT_CONTROL_RIGHT:
		case ACTION_SHIFT_CONTROL_BOTTOM:
		case ACTION_SHIFT_CONTROL_LEFT:
		case ACTION_CONTROL_TR:
		case ACTION_CONTROL_BR:
		case ACTION_CONTROL_BL:
		case ACTION_CONTROL_TL:
		case ACTION_CONTROL_CW:
		case ACTION_CONTROL_CCW:
		case ACTION_CONTROL_TOP:
		case ACTION_CONTROL_RIGHT:
		case ACTION_CONTROL_BOTTOM:
		case ACTION_CONTROL_LEFT:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				shift = (ACTION_SHIFT_CONTROL(LOWORD(wParam))) ? 1 : 0;
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) -
					((shift) ? ACTION_SHIFT_CONTROL_TR :
					ACTION_CONTROL_TR),
					shift, TRUE, FALSE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_CONTROL_ALT_TR:
		case ACTION_CONTROL_ALT_BR:
		case ACTION_CONTROL_ALT_BL:
		case ACTION_CONTROL_ALT_TL:
		case ACTION_CONTROL_ALT_CW:
		case ACTION_CONTROL_ALT_CCW:
		case ACTION_CONTROL_ALT_TOP:
		case ACTION_CONTROL_ALT_RIGHT:
		case ACTION_CONTROL_ALT_BOTTOM:
		case ACTION_CONTROL_ALT_LEFT:
		case ACTION_ALT_TR:
		case ACTION_ALT_BR:
		case ACTION_ALT_BL:
		case ACTION_ALT_TL:
		case ACTION_ALT_CW:
		case ACTION_ALT_CCW:
		case ACTION_ALT_TOP:
		case ACTION_ALT_RIGHT:
		case ACTION_ALT_BOTTOM:
		case ACTION_ALT_LEFT:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				shift = (ACTION_CONTROL_ALT(LOWORD(wParam))) ? 1 : 0;
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) -
					((shift) ? ACTION_CONTROL_ALT_TR :
					ACTION_ALT_TR),
					FALSE, shift, TRUE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_PERIOD2:
		case ACTION_PERIOD3:
		case ACTION_BOTH:
			periodModePuzzle(&widget,
				(int) LOWORD(wParam) - ACTION_PERIOD2);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SPEED:
			speedUpPuzzle(&widget);
			break;
		case ACTION_SLOW:
			slowDownPuzzle(&widget);
			break;
		case ACTION_SOUND:
			toggleSoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK | MB_ICONQUESTION);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK | MB_ICONEXCLAMATION);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK | MB_ICONINFORMATION);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) about);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(XtPointer) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (int) msg.wParam;
}

#else

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*dino.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*dino.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*dino.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*dino.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*dino.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*dino.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*dino.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*dino.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*dino.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*dino.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*dino.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*dino.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*dino.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*dino.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*dino.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*dino.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-frame", (char *) "*dino.frameColor", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*dino.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*dino.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*dino.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fn", (char *) "*dino.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*dino.font", XrmoptionSepArg, NULL},
	{(char *) "-view", (char *) "*dino.view", XrmoptionSepArg, NULL},
	{(char *) "-mode", (char *) "*dino.mode", XrmoptionSepArg, NULL},
	{(char *) "-both", (char *) "*dino.mode", XrmoptionNoArg, (char *) "4"},
	{(char *) "-orient", (char *) "*dino.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*dino.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*dino.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*dino.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-userName", (char *) "*dino.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*dino.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*dino.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*dino.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
soundCallback(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct *cbs)
{
	Boolean value = cbs->set;

	XtVaSetValues(puzzle2d,
		XtNsound, value, NULL);
}

#elif defined(HAVE_ATHENA)
static void
puzzlePracticeCancelCallback(Widget w, XtPointer clientData,
		XtPointer callData)
{
	XtPopdown(practiceDialog);
}

static void
createPracticeQuery(char *title, char *text)
{
	Widget dialog, okDialog, cancelDialog;

	practiceDialog = XtCreatePopupShell(title,
		transientShellWidgetClass, topLevel, NULL, 0);
	dialog = XtVaCreateManagedWidget("dialog",
		dialogWidgetClass, practiceDialog,
		XtNlabel, text, NULL);
	okDialog = XtVaCreateManagedWidget("OK",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(okDialog, XtNcallback,
		puzzlePracticeCallback, dialog);
	cancelDialog = XtVaCreateManagedWidget("Cancel",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(cancelDialog, XtNcallback,
		puzzlePracticeCancelCallback, dialog);
	XtRealizeWidget(practiceDialog);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(practiceDialog), &wmDeleteWindow, 1);
}

static void
puzzleRandomizeCancelCallback(Widget w, XtPointer clientData,
		XtPointer callData)
{
	XtPopdown(randomizeDialog);
}

static void
createRandomizeQuery(char *title, char *text)
{
	Widget dialog, okDialog, cancelDialog;

	randomizeDialog = XtCreatePopupShell(title,
		transientShellWidgetClass, topLevel, NULL, 0);
	dialog = XtVaCreateManagedWidget("dialog",
		dialogWidgetClass, randomizeDialog,
		XtNlabel, text, NULL);
	okDialog = XtVaCreateManagedWidget("OK",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(okDialog, XtNcallback,
		puzzleRandomizeCallback, dialog);
	cancelDialog = XtVaCreateManagedWidget("Cancel",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(cancelDialog, XtNcallback,
		puzzleRandomizeCancelCallback, dialog);
	XtRealizeWidget(randomizeDialog);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(randomizeDialog), &wmDeleteWindow, 1);
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
	XtAppContext appCon;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	Widget mainPanel, controlPanel, menuBar;
	Widget movesRowCol;
	Widget speedChangerRowCol, toggleRowCol;
	Widget playMenu;
#endif
#ifdef HAVE_MOTIF
	Widget pullDownMenu, widget;
	Widget menuBarPanel;
	Widget labelRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString, clearString;
#ifdef EXTRA
	XmString practiceString;
#endif
	XmString randomizeString, solveString;
#ifdef EXTRA
	XmString orientString;
	XmString speedString, slowString;
#endif
	XmString soundString;
#ifdef HAVE_OPENGL
	XmString viewString;
#endif
#elif defined(HAVE_ATHENA)
	XtActionsRec actions[] = {
		{(char *) "DeleteWindowProc", deleteWindowProc},
		{(char *) "ClosePopupPanel", (XtActionProc) closePopupPanel}
	};
	String fallbackResources[] = {
		(char *) "?.translations: #override <Message>WM_PROTOCOLS: DeleteWindowProc()",
		(char *) "?.TransientShell.translations: #override <Message>WM_PROTOCOLS: ClosePopupPanel()",
		NULL
	};
	Widget modeBox, speedBox;
	Widget orientBox, practiceBox;
	Widget movesLabel, recordLabel;
	Widget modeRowCol;
	Widget w;
	Widget fileLabel, playLabel, helpLabel;
	int mode, delay;
	Boolean orient, practice;
	unsigned int i;
#else
	char titleDsp[TITLE_FILE_LENGTH];
#endif

#ifdef __VMS
	int n;
	progDsp = strrchr(argv[0], ']');
	for (n = 0; progDsp[n] != '\0' && progDsp[n] != '.'; n++);
	progDsp[n] = '\0';
#else
	progDsp = strrchr(argv[0], '/');
#endif
	if (progDsp != NULL)
		progDsp++;
	if (progDsp == NULL)
		progDsp = argv[0];
	topLevel = XtVaAppInitialize(NULL, "XDino",
		options, XtNumber(options), &argc, argv,
#ifdef HAVE_ATHENA
		fallbackResources,
#else
		NULL,
#endif
		NULL);
	appCon = XtWidgetToApplicationContext(topLevel);
	if (argc != 1)
		usage(argv[0], synopsisHelp);
#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P', NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Exit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "fileMenu",
		0, fileMenuCallback,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'x', NULL, NULL, NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
#ifdef EXTRA
	practiceString = XmStringCreateSimple((char *) "Practice");
#endif
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Auto-solve");
#ifdef EXTRA
	orientString = XmStringCreateSimple((char *) "Oriented");
	speedString = XmStringCreateSimple((char *) "Speed >");
	slowString = XmStringCreateSimple((char *) "Slow <");
#endif
	soundString = XmStringCreateSimple((char *) "Sound @");
#ifdef HAVE_OPENGL
	viewString = XmStringCreateSimple((char *) "View");
#endif
	playMenu = XmVaCreateSimplePulldownMenu(menuBar, (char *) "playMenu",
		1, playMenuCallback,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
#ifdef EXTRA
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
#endif
		XmVaPUSHBUTTON, randomizeString, 'z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 's', NULL, NULL,
#ifdef EXTRA
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
#endif
		XmVaTOGGLEBUTTON, soundString, '@', NULL, NULL,
#ifdef HAVE_OPENGL
		XmVaPUSHBUTTON, viewString, 'V', NULL, NULL,
#endif
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
#ifdef EXTRA
	XmStringFree(practiceString);
#endif
	XmStringFree(randomizeString);
	XmStringFree(solveString);
#ifdef EXTRA
	XmStringFree(orientString);
	XmStringFree(speedString);
	XmStringFree(slowString);
#endif
	XmStringFree(soundString);
#ifdef HAVE_OPENGL
	XmStringFree(viewString);
#endif
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL); /* mnemonic warning on Cygwin */
	XtVaSetValues(menuBar,
		XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 0);
	widget = XtVaCreateManagedWidget("Features...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 1);
	widget = XtVaCreateManagedWidget("Synopsis...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'S', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 2);
	widget = XtVaCreateManagedWidget("References...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 3);
	widget = XtVaCreateManagedWidget("About...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 4);
	XtManageChild(menuBar);
	practiceDialog = createQuery(topLevel, "Practice Query",
		(char *) "Are you sure you want to toggle the practice mode?",
		(XtCallbackProc) puzzlePracticeCallback);
	randomizeDialog = createQuery(topLevel, "Randomize Query",
		(char *) "Are you sure you want to randomize?",
		(XtCallbackProc) puzzleRandomizeCallback);
	mainPanel = XtVaCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel, NULL);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate and out of date. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	movesText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("nullText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, " ", 2, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	recordText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	speedChangerRowCol = XtVaCreateManagedWidget("speedChangerRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(speedChangerRowCol, &speedChanger,
		(char *) speedLabel, MAX_SPEED - DEFAULT_DELAY,
		MIN_SPEED, MAX_SPEED, 2,
		(XtCallbackProc) speedChangeCallback);
#else
	createSlider(speedChangerRowCol, &speedChanger,
		(char *) speedLabel, MAX_SPEED - DEFAULT_DELAY,
		MIN_SPEED, MAX_SPEED, 2, SCROLL_SIZE,
		(XtCallbackProc) speedChangeCallback);
#endif
	labelRowCol = XtVaCreateManagedWidget("labelRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	createRadio(labelRowCol, (Widget **) &modeRadio[0],
		modeStrings, modeLabel, XtNumber(modeStrings),
		(XtCallbackProc) modeCallback);
	toggleRowCol = XtVaCreateManagedWidget("toggleRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	orientToggle = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, toggleRowCol,
		XmNset, DEFAULT_ORIENT, NULL);
	XtAddCallback(orientToggle, XmNvalueChangedCallback,
		(XtCallbackProc) orientCallback, (XtPointer) NULL);
	practiceToggle = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, toggleRowCol,
		XmNset, DEFAULT_PRACTICE, NULL);
	XtAddCallback(practiceToggle, XmNvalueChangedCallback,
		(XtCallbackProc) practiceCallback, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	message = XtVaCreateManagedWidget("Play Dino! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle2d = XtVaCreateManagedWidget("dino",
		dino2dWidgetClass, mainPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
	XtAddCallback(puzzle2d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#ifdef HAVE_OPENGL
	puzzle3d = XtVaCreateManagedWidget("dino",
		dinoGLWidgetClass, mainPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
#else
	puzzle3d = XtVaCreateManagedWidget("dino",
		dino3dWidgetClass, mainPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
#endif
#elif defined(HAVE_ATHENA)
	XtAppAddActions(appCon, actions, XtNumber(actions));
	createPracticeQuery((char *) "Practice Query",
		(char *) "Are you sure you want to toggle the practice mode?");
	createRandomizeQuery((char *) "Randomize Query",
		(char *) "Are you sure you want to randomize?");
	createHelp(topLevel, &descriptionDialog, (char *) "Description",
		(char *) descriptionHelp, (XtCallbackProc) closePopupPanel2);
	createScrollHelp(topLevel, &featuresDialog, (char *) "Features",
		(char *) featuresHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &synopsisDialog, (char *) "Synopsis",
		(char *) synopsisHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &referencesDialog, (char *) "References",
		(char *) referencesHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &aboutDialog, (char *) "About",
		(char *) aboutHelp, (XtCallbackProc) closePopupPanel2);
	mainPanel = XtVaCreateManagedWidget("form",
		panedWidgetClass, topLevel, NULL);
	menuBar = XtVaCreateManagedWidget("MenuBar",
		formWidgetClass, mainPanel,
		XtNborderWidth, 1, NULL);
	createMenu(menuBar, &fileLabel, NULL,
		fileTypes, "File", numFileTypes,
		0, False, fileMenuCallback);
	playLabel = XtVaCreateManagedWidget("Play",
		menuButtonWidgetClass, menuBar,
		XtNborderWidth, 0,
		XtNfromHoriz, fileLabel, NULL);
	playMenu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, playLabel, NULL);
	for (i = 0; i < numPlayTypes; i++) {
#ifdef HAVE_OPENGL
		if (i == numPlayTypes - 2)
#else
		if (i == numPlayTypes - 1)
#endif
		{
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu,
				XtNleftMargin, 20, NULL); /* for check */
			soundMenuItem = w;
		} else
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu, NULL);
		XtAddCallback(w,
			XtNcallback, (XtCallbackProc) playMenuCallback,
			(XtPointer) (size_t) i);
	}
	createMenu(menuBar, &helpLabel, playLabel,
		helpTypes, "Help", numHelpTypes,
		0, False, helpMenuCallback);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		formWidgetClass, mainPanel,
		XtNborderWidth, 0, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0, NULL);
	movesLabel = XtVaCreateManagedWidget("Moves",
		labelWidgetClass, movesRowCol,
		XtNborderWidth, 0, NULL);
	movesText = XtVaCreateManagedWidget("0",
		labelWidgetClass, movesRowCol,
		XtNwidth, 32,
		XtNfromHoriz, movesLabel, NULL);
	recordLabel = XtVaCreateManagedWidget("Record",
		labelWidgetClass, movesRowCol,
		XtNborderWidth, 0,
		XtNfromHoriz, movesText, NULL);
	recordText = XtVaCreateManagedWidget("0",
		labelWidgetClass, movesRowCol,
		XtNwidth, 108,
		XtNfromHoriz, recordLabel, NULL);
	speedChangerRowCol = XtVaCreateManagedWidget("speedChangerRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, movesRowCol, NULL);
	speedBox = XtVaCreateManagedWidget("speedBox",
		boxWidgetClass, speedChangerRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	modeRowCol = XtVaCreateManagedWidget("modeRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, speedChangerRowCol, NULL);
	modeBox = XtVaCreateManagedWidget("modeBox",
		boxWidgetClass, modeRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	toggleRowCol = XtVaCreateManagedWidget("toggleRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, modeRowCol, NULL);
	orientBox = XtVaCreateManagedWidget("Oriented",
		boxWidgetClass, toggleRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNstate, DEFAULT_ORIENT, NULL);
	practiceBox = XtVaCreateManagedWidget("Practice",
		boxWidgetClass, toggleRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNstate, DEFAULT_PRACTICE,
		XtNfromHoriz, orientBox, NULL);
	message = XtVaCreateManagedWidget("Welcome",
		labelWidgetClass, controlPanel,
		XtNjustify, XtJustifyLeft,
		XtNborderWidth, 0,
		XtNwidth, 180,
		XtNfromVert, toggleRowCol, NULL);
	puzzle2d = XtVaCreateManagedWidget("dino",
		dino2dWidgetClass, mainPanel,
		XtNfromVert, controlPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
	XtVaGetValues(puzzle2d,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNdelay, &delay, NULL);
	XtAddCallback(puzzle2d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#ifdef HAVE_OPENGL
	puzzle3d = XtVaCreateManagedWidget("dino",
		dinoGLWidgetClass, mainPanel,
		XtNfromVert, puzzle2d,
		XtNheight, WINDOW_HEIGHT, NULL);
#else
	puzzle3d = XtVaCreateManagedWidget("dino",
		dino3dWidgetClass, mainPanel,
		XtNfromVert, puzzle2d,
		XtNheight, WINDOW_HEIGHT, NULL);
#endif
#ifdef USE_POPUP
	createPopupMenu(modeBox, &modeNameLabel, modeStrings,
		modeLabel, 0, mode - PERIOD2,
		sizeof(modeStrings) / sizeof(*modeStrings),
		(XtCallbackProc) modeCallback);
#else
	createRadio(modeBox, (Widget **) &modeRadio[0], modeStrings,
		modeLabel, 0, mode - PERIOD2,
		sizeof(modeStrings) / sizeof(*modeStrings),
		(XtCallbackProc) modeCallback);
#endif
#ifdef USE_SPIN
	createSpinner(speedBox, &speedSliderLabel,
		speedLabel, 0, getSpeed(delay), MIN_SPEED, MAX_SPEED, True,
		speedUpCallback, speedDownCallback);
#else
	createSlider(speedBox, &speedSliderLabel, &speedSlider,
		speedLabel, 0, getSpeed(delay), MIN_SPEED, MAX_SPEED, True,
		SCROLL_SIZE, speedChangeCallback, speedChangeCallback);
#endif
	createToggle(orientBox, &orientToggle, "Oriented:", 0,
		orient, (XtCallbackProc) orientCallback);
	createToggle(practiceBox, &practiceToggle, "Practice:", 0,
		practice, (XtCallbackProc) practiceCallback);
	XtRealizeWidget(topLevel);
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
	XSetWMProtocols(XtDisplay(topLevel), XtWindow(topLevel),
		&wmDeleteWindow, 1);
#else
	puzzle2d = XtVaCreateManagedWidget("dino",
		dino2dWidgetClass, topLevel, NULL);
	XtAddCallback(puzzle2d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
	shell = XtAppCreateShell(progDsp, titleDsp,
		topLevelShellWidgetClass, XtDisplay(topLevel), NULL, 0);
#ifdef HAVE_OPENGL
	puzzle3d = XtVaCreateManagedWidget("dino",
		dinoGLWidgetClass, shell, NULL);
#else
	puzzle3d = XtVaCreateManagedWidget("dino",
		dino3dWidgetClass, shell, NULL);
#endif
#endif
	XtVaGetValues(puzzle2d,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel,
			XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
#else
	XtVaSetValues(topLevel,
		XtNinput, True,
		XtNiconPixmap, pixmap, NULL);
	XtVaSetValues(shell,
		XtNinput, True,
		XtNiconPixmap, pixmap, NULL);
#endif
	XtAddCallback(puzzle3d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#ifdef HAVE_MOTIF
	updateToggle(playMenu, &soundMenuItem,
		False,
#ifdef EXTRA
		9,
#else
		5,
#endif
		(XtCallbackProc) soundCallback);
	descriptionDialog = createHelp(menuBar, (char *) "Description",
		(char *) descriptionHelp);
	featuresDialog = createScrollHelp(menuBar, (char *) "Features",
		(char *) featuresHelp, pixmap);
	synopsisDialog = createHelp(menuBar, (char *) "Synopsis",
		(char *) synopsisHelp);
	referencesDialog = createHelp(menuBar, (char *) "References",
		(char *) referencesHelp);
	aboutDialog = createHelp(menuBar, (char *) "About",
		(char *) aboutHelp);
#endif
	initialize();
	XtRealizeWidget(topLevel);
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
	XtRealizeWidget(shell);
#endif
	XGrabButton(XtDisplay(puzzle2d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle2d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle2d),
		XCreateFontCursor(XtDisplay(puzzle2d), XC_hand2));
	XGrabButton(XtDisplay(puzzle3d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle3d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle3d),
		XCreateFontCursor(XtDisplay(puzzle3d), XC_hand2));
#ifdef HAVE_EDITRES
	XtAddEventHandler(topLevel, (EventMask) 0, TRUE,
		(XtEventHandler) _XEditResCheckMessages, NULL);
#endif
	XtAppMainLoop(appCon);

#ifdef __VMS
	return 1;
#else
	return 0;
#endif
}
#endif
