/*
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * common functions for edge detection
 */

#ifndef AVFILTER_EDGE_COMMON_H
#define AVFILTER_EDGE_COMMON_H

#include "avfilter.h"

/**
 * @brief Rounded directions used in av_image_sobel()
 */
enum AVRoundedDirection {
    DIRECTION_45UP,
    DIRECTION_45DOWN,
    DIRECTION_HORIZONTAL,
    DIRECTION_VERTICAL,
};

/**
 * Simple sobel operator to get rounded gradients
 *
 * @param w             the width of the image in pixels
 * @param h             the height of the image in pixels
 * @param dst           data pointers to magnitude image
 * @param dst_linesize  linesizes for the magnitude image
 * @param dir           data pointers to direction image
 * @param dir_linesize  linesizes for the direction image
 * @param src           data pointers to source image
 * @param src_linesize  linesizes for the source image
 */
void ff_sobel(int w, int h,
              uint16_t *dst, int dst_linesize,
              int8_t *dir, int dir_linesize,
              const uint8_t *src, int src_linesize);

/**
 * Filters rounded gradients to drop all non-maxima pixels in the magnitude image
 * Expects gradients generated by av_image_sobel()
 * Expects zero's in the destination buffer dst
 *
 * @param w             the width of the image in pixels
 * @param h             the height of the image in pixels
 * @param dst           data pointers to magnitude image
 * @param dst_linesize  linesizes for the magnitude image
 * @param dir           data pointers to direction image
 * @param dir_linesize  linesizes for the direction image
 * @param src           data pointers to source image
 * @param src_linesize  linesizes for the source image
 */
void ff_non_maximum_suppression(int w, int h,
                                uint8_t *dst, int dst_linesize,
                                const int8_t *dir, int dir_linesize,
                                const uint16_t *src, int src_linesize);

/**
 * Filters all pixels in src to keep all pixels > high,
 * and keep all pixels > low where all surrounding pixels > high
 *
 * @param low           the low threshold value
 * @param high          the hegh threshold value
 * @param w             the width of the image in pixels
 * @param h             the height of the image in pixels
 * @param dst           data pointers to destination image
 * @param dst_linesize  linesizes for the destination image
 * @param src           data pointers to source image
 * @param src_linesize  linesizes for the source image
 */
void ff_double_threshold(int low, int high, int w, int h,
                         uint8_t *dst, int dst_linesize,
                         const uint8_t *src, int src_linesize);

/**
 * Applies gaussian blur.
 * 5x5 kernels, sigma = 1.4
 *
 * @param w             the width of the image in pixels
 * @param h             the height of the image in pixels
 * @param dst           data pointers to destination image
 * @param dst_linesize  linesizes for the destination image
 * @param src           data pointers to source image
 * @param src_linesize  linesizes for the source image
 */
void ff_gaussian_blur(int w, int h,
                      uint8_t *dst, int dst_linesize,
                      const uint8_t *src, int src_linesize);

#endif
