/****************************************************************************
 *
 *   SPDX-License-Identifier: BSD-3-Clause
 *   Copyright(c) 2007-2026 Intel Corporation
 * 
 *   These contents may have been developed with support from one or more
 *   Intel-operated generative artificial intelligence solutions.
 *
 ***************************************************************************/

/**
 *****************************************************************************
 * @file dc_header_footer_lz4.h
 *
 * @ingroup Dc_DataCompression
 *
 * @description
 *      Definition of the Data Compression header and footer parameters
 *      for LZ4.
 *
 *****************************************************************************/
#ifndef DC_HEADER_FOOTER_LZ4_H_
#define DC_HEADER_FOOTER_LZ4_H_

#include "lac_common.h"

/* Header and footer size LZ4 */
#define DC_LZ4_HEADER_SIZE 7
#define DC_LZ4_FOOTER_SIZE 8

/* Values used to build the headers for LZ4 */
#define DC_LZ4_FH_ID 0x184D2204U
#define DC_LZ4_FH_FLG_VERSION 0x1
#define DC_LZ4_FH_MAX_BLK_SIZE_ENUM_MIN 4

/* Values used to build footers for LZ4 */
#define DC_LZ4_FF_END_MARK 0x0

/**
 *****************************************************************************
 * @ingroup dc_lz4_generate_header
 *      Generate the LZ4 Header.
 *
 * @description
 *      This function generates the LZ4 compression header.
 *
 * @param[in]       dest_buff        Pointer to the destination buffer the
 *                                   LZ4 header will be written to.
 * @param[in]       max_block_size   LZ4 Maximum block size.
 * @param[in]       block_indep       LZ4 block independence value.
 * @param[in,out]   count            Pointer to counter that stores
 *                                   amount of generated bytes.
 *
 * @retval CPA_STATUS_SUCCESS        Function executed successfully.
 * @retval CPA_STATUS_INVALID_PARAM  Invalid parameter passed in.
 *****************************************************************************/
CpaStatus dc_lz4_generate_header(const CpaFlatBuffer *dest_buff,
                                 const CpaDcCompLZ4BlockMaxSize max_block_size,
                                 const CpaBoolean block_indep,
                                 Cpa32U *count);

/**
 *****************************************************************************
 * @ingroup dc_lz4_generate_footer
 *      Generate the LZ4 footer.
 *
 * @description
 *      This function generates the LZ4 compression footer.
 *
 * @param[in]   dest_buff        Pointer to the destination buffer where
 *                               the LZ4 footer will be written to.
 * @param[in]   pRes             Pointer to the CpaDcRqResults structure
 *                               that holds XXHASH32.
 *
 * @retval CPA_STATUS_SUCCESS        Function executed successfully.
 * @retval CPA_STATUS_INVALID_PARAM  Invalid parameter passed in.
 *****************************************************************************/
CpaStatus dc_lz4_generate_footer(const CpaFlatBuffer *dest_buff,
                                 const CpaDcRqResults *pRes);

#endif /* DC_HEADER_FOOTER_LZ4_H_ */
