//  Copyright Takuma Yoshimura 2024.
//  Copyright Matt Borland 2024.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#define BOOST_TEST_MAIN
#define BOOST_TEST_MODULE StatsSaSpoint5Test

#ifdef BOOST_MATH_ENABLE_SYCL
#include "sycl/sycl.hpp"
#endif

#include <boost/math/tools/config.hpp>
#include <boost/test/included/unit_test.hpp>
#include <boost/test/tools/floating_point_comparison.hpp>

#include <boost/math/distributions/saspoint5.hpp>

#if __has_include(<stdfloat>)
# include <stdfloat>
#endif

using boost::math::saspoint5_distribution;

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
#include <boost/multiprecision/cpp_bin_float.hpp>
using boost::multiprecision::cpp_bin_float_quad;
#endif

template<class RealType, int N>
void do_test_saspoint5_pdf(){
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    saspoint5_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.65057384221262866484014802392420311075288403543570e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.79329640523490376041131493419821198600076403419386e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.95852304370396879516224023732198088002401888082616e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.15165745993244539388932384769132623478818505719254e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37991930003932826612228434406591616554240049257806e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.65315905048420909110038030496372707676321065455853e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.98514782971078642902580240066249560381976244868252e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.39569569206548434200616180477229996876301302607033e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.91428580496513429479068747515164587814473831035141e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.22747038893846641865142698258984886289652875016810e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.58664432176873307685856460747711890398474760309135e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.00211113907132159419276991035240896253998639909692e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.48740303942235865293078214071409846807501918299929e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.06062847902700317213276816720705695621399130112975e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.74654707859965380374478388584454386653113140787247e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.57987022193031994457557921317658430036146780413966e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.61071469126041183247373313827161939453635781053656e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.22226841107155902731102159116498022415283445125970e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.91402704796678469983705063964301389721348302634810e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.07020274682586388030767450697192728366834992164365e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.16067678880616390663660038956460612725510361666743e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.26548452861398188870931570373322249360362213948984e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38813584257594010463041381178865326769684234839162e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.53335207865128979807776513257284069794567700115319e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70762401725206223811383500786268939644546879037607e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.46875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.80837633035190650461893908063251050472045384954446e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.92009924039028830545431687288769864540254439571691e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.40625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.04458865222194329452641393707783622253522148720927e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18403802814813998631607652585109350067000954260159e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.34375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.34114858767964975455487653703370079404113295500766e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51927454673218629749392967500746624956301580234179e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.28125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72261384495736629760803695406845321684342258638651e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.95645445681747568731488283573032414811445124048278e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.08683711301287379599769504729521857091008674908403e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.21875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.22747777451211671698876953149906906288207621283740e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.37948342659965914836420291897431271791569870739942e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.54408462624872191505363112311495187968578947455635e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.72263350586688041299651258469337654343439789762344e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.15625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.91658567295250072264436343404335704809676265611623e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.12745086277094097472011430871049070050383274543942e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.35668401768623200524372663239480799018368629449958e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.60546312221207321659735213679473165482665805082365e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.09375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.87425051951658593562735199923143476116556674561953e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.16193612095139713576014676574899762241637086348666e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.46416716200748206779925127900698754118588044244443e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.77011156717390291621839083396306231100301720494794e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.03125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.05643252618763782827233685490846659295644445464474e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.27802677165093171536432271430054096075199851236277e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.36619772367581343075535053490057448137838582961826e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.27802677165093171536432271430054096075199851236277e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.03125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.05643252618763782827233685490846659295644445464474e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.77011156717390291621839083396306231100301720494794e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.46416716200748206779925127900698754118588044244443e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.16193612095139713576014676574899762241637086348666e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.09375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.87425051951658593562735199923143476116556674561953e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.60546312221207321659735213679473165482665805082365e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.35668401768623200524372663239480799018368629449958e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.12745086277094097472011430871049070050383274543942e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.15625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.91658567295250072264436343404335704809676265611623e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.72263350586688041299651258469337654343439789762344e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.54408462624872191505363112311495187968578947455635e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.37948342659965914836420291897431271791569870739942e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.21875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.22747777451211671698876953149906906288207621283740e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.08683711301287379599769504729521857091008674908403e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.95645445681747568731488283573032414811445124048278e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.28125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72261384495736629760803695406845321684342258638651e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51927454673218629749392967500746624956301580234179e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.34375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.34114858767964975455487653703370079404113295500766e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18403802814813998631607652585109350067000954260159e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.40625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.04458865222194329452641393707783622253522148720927e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.92009924039028830545431687288769864540254439571691e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.46875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.80837633035190650461893908063251050472045384954446e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70762401725206223811383500786268939644546879037607e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.53335207865128979807776513257284069794567700115319e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38813584257594010463041381178865326769684234839162e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.26548452861398188870931570373322249360362213948984e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.16067678880616390663660038956460612725510361666743e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.07020274682586388030767450697192728366834992164365e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.91402704796678469983705063964301389721348302634810e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.22226841107155902731102159116498022415283445125970e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.61071469126041183247373313827161939453635781053656e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.57987022193031994457557921317658430036146780413966e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.74654707859965380374478388584454386653113140787247e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.06062847902700317213276816720705695621399130112975e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.48740303942235865293078214071409846807501918299929e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.00211113907132159419276991035240896253998639909692e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.58664432176873307685856460747711890398474760309135e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.22747038893846641865142698258984886289652875016810e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.91428580496513429479068747515164587814473831035141e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.39569569206548434200616180477229996876301302607033e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.98514782971078642902580240066249560381976244868252e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.65315905048420909110038030496372707676321065455853e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37991930003932826612228434406591616554240049257806e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.15165745993244539388932384769132623478818505719254e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.95852304370396879516224023732198088002401888082616e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.79329640523490376041131493419821198600076403419386e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.65057384221262866484014802392420311075288403543570e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41709229627396868333284301965586098495341505333984e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.23486804023715403906392442935982382364217653266772e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08928004905124817201928007015908544910529569714342e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.70693662562544123549117032234725245555978806476399e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.72521377242204601673499906339041894791681660981780e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.90118208003638158949103509569466271651423947601700e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.20122199033087670451578082577537221533524657616689e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.60044810497290557552736366450372523266254544297541e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(9)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.62568921883700157286068393009259359194911515759460e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(10)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.87225538372111615796471446753804828223647849403478e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.27521869884978603565623102805698827447589635270965e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.79233594205638037575121942287372720508125859480601e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.39503653350126602107748523248767674409207708020015e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.06336366766043497753645671560287680725185122482037e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.78299184327161285281451711213360177146934142134357e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.54339461777955741686401041938275102206830951452701e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.15681965291001751496355126714032815564803295371462e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.85998635069315959390913560004569618374730778555584e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.62608332259613979271666391679559272984023328035102e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43781372306723685704327334057789272970358761256651e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28356349911089078230470059243188541053449454702148e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.15527095704472843921748976306796551576061214402975e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.04718107132157571106811924450948266398728667364202e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.55085695067883584460317653567009454037053097890033e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.07065059415060466523722913682664006023733154400679e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.93920671056491824666303660784084772546983031349823e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.05105752932345415433215445597062488386383300986381e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.33854958380909922933873661422920090995959934982337e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.75647800724537927733394825316181366013592737695448e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(56)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.27359757459764665859327891341356484542630879530500e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(60)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.86768631746691871635029152331878689826668170171440e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(64)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.52254909340415842254122300688608224784456787996634e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(72)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.96929220521963282370464091269016751022713088743640e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(80)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.54771289809160518824087477899271848785553979294988e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(88)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.21766390014321750130289247373333697019024113873128e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(96)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.95349215386026125880558883477711894508813402521643e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(104)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.73811374936907293362277490498742346151511716286084e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(112)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55975461659552125943580943693269482505834989982945e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(120)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41006225150819455490222552628817791206744288460867e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28296324273402304808354336538549844914883481532620e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(256)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.63180321475387230691526892287847669196997380044025e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(512)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.66189768326912251607424407981263646223230063365226e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1024)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.93708011998897739786377954224930716533616609090213e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.11452967162743112245198615236484567129951718982008e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.51482080833564338340856966996445313498079083789150e-7), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.66663174159211670027542189178751103023898387998523e-7), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.45238046039194445466823557091765258247216574897413e-8), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.34803516847185617806978815778928715951216310353916e-8), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.18523956611649861747653810989133680515605456258881e-8), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 17)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.19428394481581920558479056092124822410228982935516e-9), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.48386121491270229946980639019289494044106626390164e-9), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 19)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.24863743761365464204033370116809431054495238229389e-10), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.85627269567198410070703889816912945617135110666962e-10), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 21)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.56441321254434701892659687588069630508140502616495e-11), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.32124514546671241752437061147193743959791071682877e-11), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 23)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.20777750941928700289773286319085009877829580100394e-12), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.90212173401654051439660118394241371322264056105160e-12), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.02611352372467016891796952205325499928128114461973e-12), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.62800552193521241168029742332979605534754328327140e-13), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 27)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28273024622868202243706785685607188210679822726442e-13), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.53522776159999787175268870767580373267754737032134e-14), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 29)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.60346802350897629386145672448930505388450307742542e-14), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.66917274294807913528606004426646232913424627695599e-15), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 31)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.00436953991320274750778716874695704836278902566390e-15), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.08655220551513998962815689594959332706015306211632e-16), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 34)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.85824418029941626524148561830736428793658611772587e-17), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.10728389276235510083072832489179009106826041868534e-17), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 38)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38410697234730043795694534456878302211221269628364e-18), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.73013503193177856102694285471291153594530345945373e-19), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 42)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.16266961272612513551592941445333504133461857377298e-20), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.70333753016489787852265460543736089676535476354358e-21), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 46)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.37917223411693427580584068017775560157489239248184e-22), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.22396549352793655444565334974277876088154820156655e-23), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 50)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.27995699246102965377026386970340969776925728507708e-24), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.59994631904573126677044584818697892085174708564778e-25), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 54)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.24993294785056705165902599588868178759415543962830e-26), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 56)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.03124162154653357770314062605026847259176035189343e-26), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 58)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28905202884892491063675049720005913844624989838025e-27), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 60)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.61131503725850485091152332729977897011733755140179e-28), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 62)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.01414379732147400935170577498436447103578962194983e-29), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 64)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51767974711955685286218349809111390785355323964808e-30), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 68)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.93387460542241032622871979569574240917519314612402e-32), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 72)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.14667907118661876897809516737349031422411025903664e-34), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 76)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.60418604881272566803193825546858269428369981726767e-36), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 80)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.50065407013025533256374121391266749023560191365933e-37), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 84)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.34477198457980010827684344075634340204878641095256e-39), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 88)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.66370622590643616572397339764831428700168401193312e-41), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 92)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.72454097797900123415519497865404920881954749456838e-43), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 96)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.94459527809226549290322908699069488543719245241219e-45), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 100)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.39759301220191945453705676879390798465474357652431e-46), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 104)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18373908156550030836685714809779087770458087234145e-48), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 108)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.41209231494609468520317755439405311521523337977429e-50), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 112)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.33139424210327312273151307737011597702780857146510e-52), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 116)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.33030350328636432344828142894916983611711484679173e-54), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 120)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.30160992238849442824114913911543137689436308410131e-55), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 124)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.03376550373202254518240301652494105481741383790265e-57), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.17775859958128522725985138779564208904631966988869e-59), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.17775859958128522725985138779564208904631966988869e-59), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 136)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.75819970600899713717946709855764494008629724492550e-63), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 144)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.89409172509985281669900618169013228855150809470644e-66), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 152)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.62424737573206254076979053560233289660261329999266e-70), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 160)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.12896664446583558124263581916004539046704057066258e-73), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 168)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.75626622184041889951815556079707845048600270288339e-77), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 176)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.72916558066508520390174723482957902400087321179181e-81), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 184)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.64286269059206181735882501247406923061001590063666e-84), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 192)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.01089524070327592128619387871640956557675279607402e-88), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 200)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.79222470874823222970262177430243561283276571668161e-92), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 208)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.39067986053423638420474164411821626475827252277446e-95), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 216)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.83662075325741304737485752958570418264028678048960e-99), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 224)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.42495623858823560726925232655901309474567221057157e-102), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 232)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.47889706686580958805969806288821606452354641951732e-106), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 240)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.49340104215285543959887222384818382775213055976161e-110), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 248)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.07358423880684947255831841402543550598919723533894e-113), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 256)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.06246152052453484511308206549178590347199333203433e-117), tolerance);
}

template<class RealType, int N>
void do_test_saspoint5_cdf() {
    //
    // Basic sanity checks, tolerance is either 5 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 5;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    saspoint5_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.63772802979087199762340235165979754525757604354946e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.68073422703516098355866522837852256596174289760926e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.72758003279298484112934948051066338979626419965674e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.77889244858892026253876673752518849640459231269905e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.83545491848533781581075416061045687904524946502798e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.89826216632004296533051251749295777627320352902276e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.96860081561627764765290065134912151446894162470366e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.04817161935798121677981868384882359833404907918648e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.13928162275383718405630406427822962549133076160983e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19012422407024648274998126765803685575224367865758e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.24515968592015292591213160991283529212368507398551e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.30502444449412087831256855850010593081747948385277e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37050277816586024016699127500442336442368833061485e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.44257500572741387471454711747681504603506105875273e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.52248669635662507718372623246751628200366838868998e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.61185078990430196469782036568483593931021694091782e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.71280312689343266367958859259591543958449635451967e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.76849301027297680034801351032711397512725349083505e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.82824834763715399283433728056507088064285422281461e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.89261854625965316722646909500096608624627558333255e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.96226648191670403196061310683376239616840103866983e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.03800118279447955882543375356729458148212335628057e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.12082296340545811234455810077892380496395042488318e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.21198706286302016964660139725707925695690953535621e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.31309550000758082761278726632760757082965788779064e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.46875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.36800649890180441699589040706573844882916813862746e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.42623318957604837956085530683032551753868317236287e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.40625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.48814554885607965224431388327096964758526950639963e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.55417563843010824151990624091505209837963398189766e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.34375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.62483167142022658662698467701475815382565060700827e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.70071604494776495147661422048036562690579901259931e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.28125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.78254859359158098510888509896172431097522748703865e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.87119665000174806422420129219814480076318292452197e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.91839717977573455659490354583900768691773107778703e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.21875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.96771371640645784475148651296909803541218278211201e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.01931503128208807344707441840522746424640776733482e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.07338815387886469109799016905025356718337763965952e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.13014030144742808960599738063482904351064602625491e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.15625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.18980066583859588646492309993087622866113891170991e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.25262169970425461499760639161103103516585531687363e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.31887921568009055676985827521151927561905069756334e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.38886999325100940692217415763934459424688336515323e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.09375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.46290440538810971901075463907912052110770078557001e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.54128928252324244858536397279015277684698852998387e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.62429173448713128429248489740624277005917720602166e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.71206593195666184191280277407391829224588075252830e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.03125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.80450931389242086819408627721643501783105413645046e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.90099745314478063577240966101000818037412904585595e-1), tolerance);
    BOOST_CHECK_EQUAL(cdf(dist, static_cast<RealType>(0)), static_cast<RealType>(0.5));
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.09900254685521936422759033898999181962587095414405e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.03125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.19549068610757913180591372278356498216894586354954e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.28793406804333815808719722592608170775411924747170e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.37570826551286871570751510259375722994082279397834e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.45871071747675755141463602720984722315301147001613e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.09375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.53709559461189028098924536092087947889229921442999e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.61113000674899059307782584236065540575311663484677e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.68112078431990944323014172478848072438094930243666e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.74737830029574538500239360838896896483414468312637e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.15625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.81019933416140411353507690006912377133886108829009e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.86985969855257191039400261936517095648935397374509e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.92661184612113530890200983094974643281662236034048e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.98068496871791192655292558159477253575359223266518e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.21875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.03228628359354215524851348703090196458781721788799e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.08160282022426544340509645416099231308226892221297e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.12880334999825193577579870780185519923681707547803e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.28125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.21745140640841901489111490103827568902477251296135e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.29928395505223504852338577951963437309420098740069e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.34375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.37516832857977341337301532298524184617434939299173e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.44582436156989175848009375908494790162036601810234e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.40625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.51185445114392034775568611672903035241473049360037e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.57376681042395162043914469316967448246131682763713e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.46875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.63199350109819558300410959293426155117083186137254e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.68690449999241917238721273367239242917034211220936e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.78801293713697983035339860274292074304309046464379e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.87917703659454188765544189922107619503604957511682e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.96199881720552044117456624643270541851787664371943e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.03773351808329596803938689316623760383159896133017e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.10738145374034683277353090499903391375372441666745e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.17175165236284600716566271943492911935714577718539e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.23150698972702319965198648967288602487274650916495e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.28719687310656733632041140740408456041550364548033e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.38814921009569803530217963431516406068978305908218e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.47751330364337492281627376753248371799633161131002e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.55742499427258612528545288252318495396493894124727e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.62949722183413975983300872499557663557631166938515e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.69497555550587912168743144149989406918252051614723e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.75484031407984707408786839008716470787631492601449e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.80987577592975351725001873234196314424775632134242e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.86071837724616281594369593572177037450866923839017e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.95182838064201878322018131615117640166595092081352e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.03139918438372235234709934865087848553105837529634e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.10173783367995703466948748250704222372679647097724e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.16454508151466218418924583938954312095475053497202e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.22110755141107973746123326247481150359540768730095e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.27241996720701515887065051948933661020373580034326e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.31926577296483901644133477162147743403825710239074e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.36227197020912800237659764834020245474242395645054e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.43871020523024507614851721583135267180360280594143e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.50483092818015575884212092733928976209364127755594e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.56280494417603102056412672973803628491156640138235e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.61420731826426116856340074125786936047174397075840e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.66021349395697519291462973006173132761227963124588e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.70172157747564156568159926049840862439194389470164e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.73943162478672589326613255747804240803057053984280e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.77389877435125713385213180379500898219171021326151e-1), tolerance);
}

template<class RealType, int N>
void do_test_saspoint5_ccdf() {
    //
    // Basic sanity checks, tolerance is either 5 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 5;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    saspoint5_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-2))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.86071837724616281594369593572177037450866923839017e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.80987577592975351725001873234196314424775632134242e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.75484031407984707408786839008716470787631492601449e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.69497555550587912168743144149989406918252051614723e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.62949722183413975983300872499557663557631166938515e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.55742499427258612528545288252318495396493894124727e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.47751330364337492281627376753248371799633161131002e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.38814921009569803530217963431516406068978305908218e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.28719687310656733632041140740408456041550364548033e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.9375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.23150698972702319965198648967288602487274650916495e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.17175165236284600716566271943492911935714577718539e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.8125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.10738145374034683277353090499903391375372441666745e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.03773351808329596803938689316623760383159896133017e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.6875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.96199881720552044117456624643270541851787664371943e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.87917703659454188765544189922107619503604957511682e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.5625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.78801293713697983035339860274292074304309046464379e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.68690449999241917238721273367239242917034211220936e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.46875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.63199350109819558300410959293426155117083186137254e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.57376681042395162043914469316967448246131682763713e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.40625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.51185445114392034775568611672903035241473049360037e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.44582436156989175848009375908494790162036601810234e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.34375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.37516832857977341337301532298524184617434939299173e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.29928395505223504852338577951963437309420098740069e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.28125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.21745140640841901489111490103827568902477251296135e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.12880334999825193577579870780185519923681707547803e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.234375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.08160282022426544340509645416099231308226892221297e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.21875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.03228628359354215524851348703090196458781721788799e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.203125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.98068496871791192655292558159477253575359223266518e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.92661184612113530890200983094974643281662236034048e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.171875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.86985969855257191039400261936517095648935397374509e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.15625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.81019933416140411353507690006912377133886108829009e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.140625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.74737830029574538500239360838896896483414468312637e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.68112078431990944323014172478848072438094930243666e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.109375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.61113000674899059307782584236065540575311663484677e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.09375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.53709559461189028098924536092087947889229921442999e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.078125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.45871071747675755141463602720984722315301147001613e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.0625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.37570826551286871570751510259375722994082279397834e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.046875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.28793406804333815808719722592608170775411924747170e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.03125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.19549068610757913180591372278356498216894586354954e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.015625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.09900254685521936422759033898999181962587095414405e-1), tolerance);
    BOOST_CHECK_EQUAL(cdf(dist, static_cast<RealType>(0)), static_cast<RealType>(0.5));
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.015625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.90099745314478063577240966101000818037412904585595e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.03125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.80450931389242086819408627721643501783105413645046e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.046875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.71206593195666184191280277407391829224588075252830e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.0625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.62429173448713128429248489740624277005917720602166e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.078125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.54128928252324244858536397279015277684698852998387e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.09375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.46290440538810971901075463907912052110770078557001e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.109375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.38886999325100940692217415763934459424688336515323e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.31887921568009055676985827521151927561905069756334e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.140625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.25262169970425461499760639161103103516585531687363e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.15625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.18980066583859588646492309993087622866113891170991e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.171875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.13014030144742808960599738063482904351064602625491e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.07338815387886469109799016905025356718337763965952e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.203125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.01931503128208807344707441840522746424640776733482e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.21875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.96771371640645784475148651296909803541218278211201e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.234375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.91839717977573455659490354583900768691773107778703e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.87119665000174806422420129219814480076318292452197e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.28125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.78254859359158098510888509896172431097522748703865e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.70071604494776495147661422048036562690579901259931e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.34375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.62483167142022658662698467701475815382565060700827e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.55417563843010824151990624091505209837963398189766e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.40625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.48814554885607965224431388327096964758526950639963e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.42623318957604837956085530683032551753868317236287e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.46875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.36800649890180441699589040706573844882916813862746e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.31309550000758082761278726632760757082965788779064e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.5625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.21198706286302016964660139725707925695690953535621e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.12082296340545811234455810077892380496395042488318e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.6875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.03800118279447955882543375356729458148212335628057e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.96226648191670403196061310683376239616840103866983e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.8125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.89261854625965316722646909500096608624627558333255e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.82824834763715399283433728056507088064285422281461e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.9375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.76849301027297680034801351032711397512725349083505e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.71280312689343266367958859259591543958449635451967e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.61185078990430196469782036568483593931021694091782e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.52248669635662507718372623246751628200366838868998e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.44257500572741387471454711747681504603506105875273e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37050277816586024016699127500442336442368833061485e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.30502444449412087831256855850010593081747948385277e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.24515968592015292591213160991283529212368507398551e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19012422407024648274998126765803685575224367865758e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.13928162275383718405630406427822962549133076160983e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.04817161935798121677981868384882359833404907918648e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.96860081561627764765290065134912151446894162470366e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.89826216632004296533051251749295777627320352902276e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.83545491848533781581075416061045687904524946502798e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.77889244858892026253876673752518849640459231269905e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.72758003279298484112934948051066338979626419965674e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.68073422703516098355866522837852256596174289760926e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(4))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.63772802979087199762340235165979754525757604354946e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(4.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.56128979476975492385148278416864732819639719405857e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.49516907181984424115787907266071023790635872244406e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(5.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43719505582396897943587327026196371508843359861765e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(6))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38579268173573883143659925874213063952825602924160e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(6.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.33978650604302480708537026993826867238772036875412e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(7))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.29827842252435843431840073950159137560805610529836e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(7.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.26056837521327410673386744252195759196942946015720e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(8))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.22610122564874286614786819620499101780828978673850e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(9))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.16519065397529967010603916749661994058495038701903e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(10))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.11285389689081092090080900757256655213056507935765e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(11))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.06722664628240641657463412013080864534542465320880e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.02697048057896404675748265840787538239835533463915e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(13))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.91095580948192894243059119812739170064298855954150e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.58851606064421509151228732938834025845710632753865e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(15))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.29657717833196446423749763633852786438048712229891e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.03056141356415128156562790092782169373827214514422e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(18))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.56233191903709448828683239881723764461808662475751e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.16188637156340219345609662100276117883253664060901e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(22))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.81416233095397134117651222065771104114974953207875e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.50842302656587649324679534871796140718152459081608e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(26))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.23677655844367661796208750471724232185503310585430e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.99327211922253387063577514052912042212460501121145e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(30))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.77332469896087134873409606852930319306153102107748e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.57333571766941514095434647381791053890536948900264e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(36))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.22230201122276543582018268125914307770458422539453e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.92306196750274052986046861933822660319647411503460e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(44))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.66393766871935144778495176368042799313533760396735e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.43664512355365430998504793608427289643995825810352e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(52))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.23512057902480812783067842173364430254180391574517e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.05480824780342286194892111115606787981112335506282e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(60))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.89220914007246019306457129229565668476439227449653e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.74458482861654721048545411630851774084511858223315e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(72))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.48596753801598503319767014729339663622931424118435e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.26600858175294196667322269091614319302918711248774e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(88))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.07590501853150263567490077970815440178521221113294e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.90943298811551585430796926339912286222583383628380e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(104))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.76204970249877025738751243150107820229031119377257e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.63035057455899788326457066144517427412403198177255e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(120))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.51172649975372596610965298504076329424502797908092e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.40413948757894194852726127978788270443438529203793e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.43203077043778334466271831424077221758122536074329e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(512))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.73229441904414627006839605354679934626132938387870e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1024))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.23125360026260788506665554837055341098621632464561e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 11))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.73810855742053432677942037710993481024582345070051e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.19474376202028990304398162068685909212949561120262e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 13))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.38834786517841980543748022239405103577545566369058e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.10703837128619218169124143528829380601241157169895e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 15))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19901414203078529527146967144846039577487804212104e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55594175267868082494169767822936567533407971376816e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 17))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.10071922510385832628139939726551776872500893306030e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 18))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.78577261178721830451003518231079692441875499186244e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 19))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.50662913793980803259595258646776816982578828753293e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.89440319684962022373915418699465975439163763242915e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 21))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.75407315047618220330559697044875151329985165407760e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 22))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.94758093727122517797618473870421858045294374184298e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 23))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.37722626166484277233388378972371910745163900023301e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.73885317860649699113798763978865018837655503068780e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.88660557570383272842403146191017664814974769956447e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 26))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.86966373050763507830869601193566593204764987942763e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 27))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.44342136111091391837428138940493220527760482550819e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.43489115282263847728003495903341128041631655237825e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 29))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.72174032467196796417055336934387675311978746399112e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 30))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21746039858699001180528503217054534969400689600469e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 31))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.60877573465331935753972340571396643508918647400950e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.08733904872845891588154720869252063263709578283331e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 34))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.04367878835689816266760026397728945536920490441769e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 36))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52184171018215269383027201178304809813110038062619e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 38))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.60921434092694304578619454290395143640352517265958e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.80460861796838148522601893634284307293180095878460e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 42))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90230467086052636671801670850090200893179111799503e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 44))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.51152425899360606075482273396798501430313790237214e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 46))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.75576235566956348303137380307065627552780450075955e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37788123437797396666202471539368169920940171968838e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 50))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.18894063132478530361545271176786170012121225204887e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 52))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.94470319196342264878568431977547823579033279922390e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 54))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.97235160481658539831961233459162678510644673997599e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.48617580461701122279770683403051652704243983949025e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 58))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.43087902860685242952854599081275466963995418723234e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 60))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.71543951568387529445493346979934712370552753482953e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 62))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.85771975818704991725083904340140685678094729181896e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.28859879179803026143850997564073108091243061664589e-11), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 68))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.32214969811127894197045507227361880900936362117960e-11), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 72))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.80537424537930446531251492646463803649696387208569e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 76))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.45134356135114531072734934087281275202470211834375e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.62835890338181277331789748787296614195690507700714e-13), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 84))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.07089725845700036860694806306823770374585029986489e-14), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 88))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.26772431461440436935874981470647001703932928470872e-14), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 92))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.66931078653610734665125754039671239552191451228014e-15), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41732769663403286311621332289317114802389526570992e-15), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 100))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.54331924158508592432390764336465547336588796613693e-16), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 104))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.85829810396271716489312806849560623979223766702760e-17), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 108))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.21457452599067943835349195212917512291401637194850e-17), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.53643631497669868784011108970123778268444560432089e-18), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 116))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38410907874417467770730159801145325661085046788106e-18), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 120))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.46027269686043669786030013601997303312653750436687e-19), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 124))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.65068174215109174689577917816952001659129997288134e-20), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.16267043553777293686425909692485421878259294596496e-20), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.16267043553777293686425909692485421878259294596496e-20), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 136))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.35166902221110808556756707276211088178619562345223e-21), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 144))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.44793138881942553480799933647572308735410707938645e-23), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 152))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.27995711801214095925541775450484763569750411131894e-24), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 160))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.29997319875758809953465243130019854989301769477462e-25), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 168))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.06248324922349256220915840763819709280743596647028e-26), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 176))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28905203076468285138072402969870025328294044417034e-27), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 184))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.05657519227926782112952519535313715734583596556713e-29), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 192))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.03535949517454238820595324747603340202581381400690e-30), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 200))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.14709968448408899262872077968737723090225341248585e-31), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 208))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.96693730280255562039295048730519109566688181166036e-32), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 216))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.22933581425159726274559405456576710378996415685244e-33), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 224))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.68334883907248289215996284103605325376468341179834e-35), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 232))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.80209302442030180759997677564753362950438836016578e-36), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 240))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.00130814026268862974998548477970853195201855431423e-37), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 248))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.87581758766418039359374092798731783299781533977493e-38), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.17238599229011274599608807999207364564425192108311e-39), tolerance);
}

template<class RealType, int N>
void do_test_saspoint5_quantile_nearzero() {
    //
    // Basic sanity checks, tolerance is either 4 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 4;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    saspoint5_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.03125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.52796721097108753422708089760626414214332697170320e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.56591342761460650504994018321276991271746451691815e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.09375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.47218526924747883249737601803820876470332879082284e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -7.64871492892195438064623231330402224448304864505255e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.15625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.49147047287704191198103513188406591369158240312651e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.83944503273842706198239703540854699099743899944658e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.21875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.88121977453918311545713678336944882698543050262508e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.28383277518932774280834618691095083936296378494548e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.28125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.91036143728268221909726623759154622815926077224027e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.21997830440617027092737461125926863143124423787158e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.34375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.31665804788247227155363433182318290115115268840234e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.93390030300024509983060478023436359505370694478274e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.40625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.90586867715715827365143687189250571789458607638023e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.12403044179836064404720613683569098344014116798151e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.46875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.11633396058323297953641398479658668551534240629654e-2), tolerance);
    BOOST_CHECK_EQUAL(quantile(dist, static_cast<RealType>(0.5)), static_cast<RealType>(0));
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.50390625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.14052785123109810231692510348602868600338388040310e-3), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.23083326042460532375320968182992121472721389674838e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.51171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.85291421003937677871303076502465267120545270922005e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.515625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.48262870167331460723959759110260709264416995145257e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.51953125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.12205802380990657930467930989682569659575359795940e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.77305954522603022893044033304410291628163249182374e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.52734375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.43730759016907991505397132799991250199870494073045e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.53125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.11633396058323297953641398479658668551534240629654e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.53515625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.81156327963950319747789851203804452183342931095156e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5390625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.52434225749394951914392360992042907273807357938464e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.54296875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.25596345497536099309066715771745797836543079457304e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.546875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.00768452063066360042362619764345937296973490700293e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.55078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.78074386337778547450756599814721014853736307848400e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5546875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.57637357991828558350746829889151355918095876787015e-2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.55859375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.03958102936522630889618724151593050682836910322233e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.12403044179836064404720613683569098344014116798151e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.56640625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21111282406179691889755301310617804156919563608730e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5703125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.30095831353205038001599004861145346134019254008840e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.57421875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.39370061391699662824182194486817335276363125501648e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.578125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.48947760819317620418221654282830443278848716355952e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.58203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.58843194157470428889620974461757508836357970580066e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5859375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.69071158647072305786338619311971877757990751767576e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.58984375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.79647039926214909876835259387875129672714154918583e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.59375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90586867715715827365143687189250571789458607638023e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.59765625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.01907372223400262964766551738092686471437958693421e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.13626041895044652657368022843673732366221034591882e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.60546875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.25761183081873425453317860727230431668344830552276e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.609375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.38331982156037948249435913385386178518604561255635e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.61328125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51358570582737493855890017214322139090955378416415e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.64862093447749446836087591919859178171321397783436e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.62109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.78864781940101196225997767818887450385653350744028e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.93390030300024509983060478023436359505370694478274e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.62890625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.08462477761234306446751675008291773968116735262918e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6328125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.24108096043381713255460633677010126705310806557902e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.63671875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.40354282984953644763765300071916088017186215793874e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.640625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.57229962948879265758087216105113103390988022371525e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.64453125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.74765694682819607503814996847853363329594489637064e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6484375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.92993787373911729744782959203924101663354892145487e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.65234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.11948425704142844251280879133218574177911117813611e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.65625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.31665804788247227155363433182318290115115268840234e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.66015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.52184275961932281071152630025400546639852185492776e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6640625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.73544504485420224870167294387230375351476906471546e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.66796875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.95789640337012040028773901682608788684052806162084e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.671875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.18965503395137956448285166074873893264110804474852e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.67578125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.43120784446902465897667277460048286927884165223763e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6796875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.68307263618494854564544144712177429500269729880351e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.68359375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.94580048000377446115526240745459082802949048053747e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.21997830440617027092737461125926863143124423787158e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.69140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.50623171706258215659862497663103390044283888606793e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6953125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.80522808468923275824853139897350453182651092750420e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.69921875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.11767989861122001397308542198440114248503523449816e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.703125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.44434845679017252964098546334886010522317885397184e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.70703125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.78604789681373771559537778013045202059314559478553e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.14364961859827390982162302693689202778040961541097e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.71484375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.51808714040276392410190679405041748405632099066392e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.71875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.91036143728268221909726623759154622815926077224027e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.72265625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.32154681743453457545426144342359967492966930439508e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7265625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.75279739912087792579527536186650514912572075425034e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.73046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.02053542591698211050464719677242867419786774375833e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.734375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.06805533335867623214314225100786133910606465280065e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.73828125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.11798341618055383937616799179262778846061718089667e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7421875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.17047495787852935040070049482482614635865914736031e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.74609375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.22569764738178929583883189036464601993502267035808e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28383277518932774280834618691095083936296378494548e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.75390625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.34507656531879179894649466860868632699377270813348e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7578125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40964166091862115574755023135179331832042209281055e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.76171875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.47775878407059212254277747469428558567366208287249e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.765625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.54967859343857147744530902523828155595548983145552e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.76953125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.62567376708606422065437126777269201178590320302836e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7734375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70604134209343637939962724226064374287142808011846e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.77734375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.79110534768099925573827334628551158224393429767853e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.78125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.88121977453918311545713678336944882698543050262508e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.78515625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.97677193016829061858313922289253621707890698703331e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.7890625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.07818623846627168681622781867868137142587154742683e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.79296875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18592855185448213952914641315720989891399211112618e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.796875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.30051105624503972344131618629751440966474716683287e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.80078125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.42249786355837299731799089007913700561632679250929e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.55251140382947389347453508427822893893843094122006e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.80859375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.69123974986228153051487214813181004658778348371858e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.83944503273842706198239703540854699099743899944658e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.81640625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.99797313731027706223638793579547090613393780037951e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.16776490443789287489784195960321745577830473993117e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.82421875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.34986911285371709243134965339463237363883153982305e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.828125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.54545757031320957733016491289957363442390624403356e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.83203125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.75584271388037850609173432108976499052095906002965e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8359375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.98249820637230758503593250844352437505576086329541e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.83984375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.22708312477451358281191586486620734769643769191868e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.84375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.49147047287704191198103513188406591369158240312651e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.84765625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.77778092239392917026904003217947497405149113745180e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8515625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.08842290488213149257806983557860643076120157707468e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.85546875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.42614045478507551744169142637868564301201400254633e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.859375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.79407056051931174009446309499422812782537696684469e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.86328125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.19581224082198008577426461565974789703165385453255e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8671875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.63551016172006234258563545412611777992137495354692e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.87109375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.11795639223606059741678567601858859535625176228939e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.64871492892195438064623231330402224448304864505255e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.87890625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.23427499059001976769144078614154782774804457875826e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8828125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.88224092195729153863789585576784499265806702821498e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.88671875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.60156902856277721120026862113617294726340622812143e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.890625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.04028650542151264864039545865194654420659149847715e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.89453125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.12987606827387901549180798300970689324385087503567e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8984375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.23043939515379786765482759320371211340959481379476e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.90234375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.34380276304930757031534891808152959303504163146383e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.90625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.47218526924747883249737601803820876470332879082284e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.91015625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.61830428929231043118455947432627297107033412348157e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.78551541642651806413112158903680754166702207111037e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.91796875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.97800037388343052469605081868946702260361591387557e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.921875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.20102262690774549296442657868648686186513838548331e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.92578125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.46128002888129351171085541318961326325857070786293e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9296875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.76739893404884608939800784860050078072327340444416e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.93359375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.13063845566677402478557805956745594172754985803934e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.56591342761460650504994018321276991271746451691815e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.94140625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.09331197973287229185946564870900349250152411988374e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9453125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.74040079933008774660558150020826896333899909298154e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.94921875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.54582183342592176371744225018671216049948426069910e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.953125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.56507752301931033036206963972726950780990330962767e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.95703125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.88016814096703401470072036376924072379132872310869e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9609375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.61631402738793734570515994292749381948933242893841e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.96484375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.19723988112353634841352379780826428736159712449728e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.96875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52796721097108753422708089760626414214332697170320e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.97265625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.01231735395463433145333990641210588531111270293944e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9765625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.76159370958398793120791299241979313515017333512222e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.98046875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.00925470915587723915099888946681205221515647834867e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.984375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.31534559333932129219229424636026382581685146602733e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.98828125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.13177617125304414676334752220512104772421667697176e3), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9921875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56685855095635428319020462289713380105568378947326e3), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.99609375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.03488985899360430847226178216145298481879311104883e4), tolerance);
}

template<class RealType, int N>
void do_test_saspoint5_quantile_lower() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    saspoint5_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -3)), BOOST_MATH_BIG_CONSTANT(RealType, N, -7.64871492892195438064623231330402224448304864505255e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -4)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.56591342761460650504994018321276991271746451691815e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -5)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.52796721097108753422708089760626414214332697170320e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -6)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.31534559333932129219229424636026382581685146602733e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -7)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.56685855095635428319020462289713380105568378947326e3), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -8)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.03488985899360430847226178216145298481879311104883e4), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -10)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.66560111810464968102166352094329039797553809013180e5), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -12)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.66887306794201818857999976857911823063870328368667e6), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -14)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.27176145418643796488192673589719523675773715325372e7), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -16)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.83544414827242601690013182749435284100680743652971e8), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -20)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.74992376775466783667682507089110333268769521176506e11), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -24)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.47981285598233127388251456426050883440628660913084e13), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -28)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.14683221929996578514587273759806303045535542474016e16), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -32)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.93589050191487067535089237582921756933019831739543e18), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -40)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.92406520022739246958562710299618130085960993793414e23), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -48)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.26095536962330863369654555704556352597738045353834e28), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -56)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.26379711036337395023730103376706642065822106379442e32), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -64)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.41576207424774090175793327287164715152292673169682e37), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -80)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.32605209918111709774537445734765990902394758994645e47), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -96)), BOOST_MATH_BIG_CONSTANT(RealType, N, -9.99031769477504631556283515692397586421188586846623e56), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -112)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.29080877757089340022276728331106401823387140732677e66), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -128)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.84288833730567254754590245964397936517161760904520e76), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -160)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.39952895147018642535639799349517299692324064545541e95), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -192)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.27102405389367073393026479530690706073270941007595e114), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -224)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.15679975802253118434756797269580371455350883468455e134), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -256)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.13391890807707704630144658385551537641074743396505e153), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -256)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.13391890807707704630144658385551537641074743396505e153), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -320)), BOOST_MATH_BIG_CONSTANT(RealType, N, -7.26134976857812288039693249651849278418162312282066e191), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -384)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.47090928629257240812467048839513197145079590955310e230), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -448)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.40806860386563308738645719151345111937971699513862e268), tolerance);
}

template<class RealType, int N>
void do_test_saspoint5_quantile_upper() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    saspoint5_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -3))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.64871492892195438064623231330402224448304864505255e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -4))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.56591342761460650504994018321276991271746451691815e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52796721097108753422708089760626414214332697170320e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -6))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.31534559333932129219229424636026382581685146602733e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -7))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56685855095635428319020462289713380105568378947326e3), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -8))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.03488985899360430847226178216145298481879311104883e4), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -10))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.66560111810464968102166352094329039797553809013180e5), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.66887306794201818857999976857911823063870328368667e6), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.27176145418643796488192673589719523675773715325372e7), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.83544414827242601690013182749435284100680743652971e8), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.74992376775466783667682507089110333268769521176506e11), tolerance);
}

template<class RealType, int N>
void do_test_saspoint5_locscale_param() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //

    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    saspoint5_distribution<RealType> dist_0_1(static_cast<RealType>(0), static_cast<RealType>(1));
    saspoint5_distribution<RealType> dist_1_3(static_cast<RealType>(1), static_cast<RealType>(3));

    BOOST_CHECK_CLOSE(entropy(dist_0_1), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.6399244456803064957308496039071853510), tolerance);
    BOOST_CHECK_CLOSE(entropy(dist_1_3), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.6399244456803064957308496039071853510) + log(static_cast<RealType>(3)), tolerance);

    BOOST_CHECK_EQUAL(median(dist_0_1), static_cast<RealType>(0));
    BOOST_CHECK_EQUAL(median(dist_1_3), static_cast<RealType>(1));

    BOOST_CHECK_EQUAL(mode(dist_0_1), static_cast<RealType>(0));
    BOOST_CHECK_EQUAL(mode(dist_1_3), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(pdf(dist_0_1, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.61071469126041183247373313827161939453635781053656e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist_1_3, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.36619772367581343075535053490057448137838582961826e-1) / 3, tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.86071837724616281594369593572177037450866923839017e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.86071837724616281594369593572177037450866923839017e-1), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, quantile(dist_0_1, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, quantile(dist_1_3, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, quantile(dist_0_1, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, quantile(dist_1_3, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
}

BOOST_AUTO_TEST_CASE(saspoint5_pdf_fp64)
{
    do_test_saspoint5_pdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(saspoint5_pdf_std64)
{
    do_test_saspoint5_pdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(saspoint5_pdf_fp128)
{
    do_test_saspoint5_pdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(saspoint5_cdf_fp64)
{
    do_test_saspoint5_cdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(saspoint5_cdf_std64)
{
    do_test_saspoint5_cdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(saspoint5_cdf_fp128)
{
    do_test_saspoint5_cdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(saspoint5_ccdf_fp64)
{
    do_test_saspoint5_ccdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(saspoint5_ccdf_std64)
{
    do_test_saspoint5_ccdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(saspoint5_ccdf_fp128)
{
    do_test_saspoint5_ccdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(saspoint5_quantile_nearzero_fp64)
{
    do_test_saspoint5_quantile_nearzero<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(saspoint5_quantile_nearzero_std64)
{
    do_test_saspoint5_quantile_nearzero<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(saspoint5_quantile_nearzero_fp128)
{
    do_test_saspoint5_quantile_nearzero<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(saspoint5_quantile_lower_fp64)
{
    do_test_saspoint5_quantile_lower<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(saspoint5_quantile_lower_std64)
{
    do_test_saspoint5_quantile_lower<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(saspoint5_quantile_lower_fp128)
{
    do_test_saspoint5_quantile_lower<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(saspoint5_quantile_upper_fp64)
{
    do_test_saspoint5_quantile_upper<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(saspoint5_quantile_upper_std64)
{
    do_test_saspoint5_quantile_upper<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(saspoint5_quantile_upper_fp128)
{
    do_test_saspoint5_quantile_upper<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(saspoint5_locscale_fp64)
{
    do_test_saspoint5_locscale_param<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(saspoint5_locscale_std64)
{
    do_test_saspoint5_locscale_param<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(saspoint5_locscale_fp128)
{
    do_test_saspoint5_locscale_param<cpp_bin_float_quad, 113>();
}
#endif
