// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright (C) 2024 Renesas Electronics Corp.
 */

#include <asm/io.h>
#include <dm.h>
#include <errno.h>
#include <hang.h>
#include <linux/sizes.h>
#include <ram.h>
#include "dbsc5.h"

/* AXMM */
#define AXMM_ADSPLCR0				0x4008
#define AXMM_ADSPLCR1				0x400C
#define AXMM_ADSPLCR2				0x4010
#define AXMM_ADSPLCR3				0x4014
#define AXMM_MMCR				0x4300
#define AXMM_TR0CR0				0x51000
#define AXMM_TR1CR0				0x51004
#define AXMM_TR2CR0				0x51008
#define AXMM_TR3CR				0x5100C
#define AXMM_TR3CR0				0x5100C
#define AXMM_TR0CR1				0x51100
#define AXMM_TR1CR1				0x51104
#define AXMM_TR2CR1				0x51108
#define AXMM_TR3CR1				0x5110C
#define AXMM_TR0CR2				0x51200
#define AXMM_TR1CR2				0x51204
#define AXMM_TR2CR2				0x51208
#define AXMM_TR3CR2				0x5120C

#define ACTEXT_RT0_R				0xFFC50800
#define ACTEXT_RT0_W				0xFFC51800
#define ACTEXT_IR0_R				0xFF890800
#define ACTEXT_IR0_W				0xFF891800
#define ACTEXT_IR1_R				0xFF892800
#define ACTEXT_IR1_W				0xFF893800
#define SI0_RW_MAX				0xF1201110
#define SI1_RW_MAX				0xF1202110

/* DBSC */
#define DBSC_A_CH_OFFSET			0x8000
#define DBSC_D_CH_OFFSET			0x4000

#define DBSC_SYSCNT0				0x0100
#define DBSC_SYSCNT1				0x0104
#define DBSC_FCPRSCTRL				0x0110
#define DBSC_DBBUS0CNF2				0x0808
#define DBSC_DBCAM0CNF1				0x0904
#define DBSC_DBCAM0CNF2				0x0908
#define DBSC_DBCAM0CNF3				0x090C
#define DBSC_DBCAMDIS				0x09FC
#define DBSC_DBSCHCNT0				0x1000
#define DBSC_DBSCHSZ0				0x1010
#define DBSC_DBSCHRW0				0x1020
#define DBSC_SCFCTST2				0x1048
#define DBSC_DBSCHQOS_0_0			0x1100
#define DBSC_DBSCHQOS_0_1			0x1104
#define DBSC_DBSCHQOS_0_2			0x1108
#define DBSC_DBSCHQOS_0_3			0x110C
#define DBSC_DBSCHQOS_4_0			0x1140
#define DBSC_DBSCHQOS_4_1			0x1144
#define DBSC_DBSCHQOS_4_2			0x1148
#define DBSC_DBSCHQOS_4_3			0x114C
#define DBSC_DBSCHQOS_9_0			0x1190
#define DBSC_DBSCHQOS_9_1			0x1194
#define DBSC_DBSCHQOS_9_2			0x1198
#define DBSC_DBSCHQOS_9_3			0x119C
#define DBSC_DBSCHQOS_12_0			0x11C0
#define DBSC_DBSCHQOS_12_1			0x11C4
#define DBSC_DBSCHQOS_12_2			0x11C8
#define DBSC_DBSCHQOS_12_3			0x11CC
#define DBSC_DBSCHQOS_13_0			0x11D0
#define DBSC_DBSCHQOS_13_1			0x11D4
#define DBSC_DBSCHQOS_13_2			0x11D8
#define DBSC_DBSCHQOS_13_3			0x11DC
#define DBSC_DBSCHQOS_14_0			0x11E0
#define DBSC_DBSCHQOS_14_1			0x11E4
#define DBSC_DBSCHQOS_14_2			0x11E8
#define DBSC_DBSCHQOS_14_3			0x11EC
#define DBSC_DBSCHQOS_15_0			0x11F0
#define DBSC_DBSCHQOS_15_1			0x11F4
#define DBSC_DBSCHQOS_15_2			0x11F8
#define DBSC_DBSCHQOS_15_3			0x11FC

/* CCI */
#define CCIQOS00				0xC020
#define CCIQOS01				0xC024
#define CCIQOS10				0xD000
#define CCIQOS11				0xD004

/* QOS */
#define QOS_FIX_QOS_BANK0			0x0
#define QOS_FIX_QOS_BANK1			0x1000
#define QOS_BE_QOS_BANK0			0x2000
#define QOS_BE_QOS_BANK1			0x3000
#define QOS_SL_INIT				0x8000
#define QOS_REF_ARS				0x8004
#define QOS_STATQC				0x8008
#define QOS_REF_ENBL				0x8044
#define QOS_BWG					0x804C
#define QOS_RAS					0x10000
#define QOS_FSS					0x10048
#define QOS_RAEN				0x10018
#define QOS_DANN_LOW				0x10030
#define QOS_DANN_HIGH				0x10034
#define QOS_DANT				0x10038
#define QOS_EMS_LOW				0x10040
#define QOS_EMS_HIGH				0x10044
#define QOS_INSFC				0x10050
#define QOS_EARLYR				0x10060
#define QOS_RACNT0				0x10080
#define QOS_STATGEN0				0x10088

#define QOSWT_FIX_QOS_BANK0			0x800
#define QOSWT_FIX_QOS_BANK1			0x1800
#define QOSWT_BE_QOS_BANK0			0x2800
#define QOSWT_BE_QOS_BANK1			0x3800
#define QOSWT_WTEN				0x8030
#define QOSWT_WTREF				0x8034
#define QOSWT_WTSET0				0x8038
#define QOSWT_WTSET1				0x803C

static const struct {
	u64	fix;
	u64	be;
} g_qosbw_tbl[] = {
	{ 0x000C04010000FFFF, 0x00200030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00200030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00200030004FFC01 },
	{ 0x000C04010000FFFF, 0x0000000000000000 },
	{ 0x000C04080000FFFF, 0x00200030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00200030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04100000FFFF, 0x00100030004FFC01 },
	{ 0x000C04100000FFFF, 0x00100030004FFC01 },
	{ 0x000C04100000FFFF, 0x0000000000000000 },
	{ 0x000C08140000FFFF, 0x00100030004FFC01 },
	{ 0x000C08140000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFF0, 0x0000000000000000 },
	{ 0x000C04100000FFFF, 0x00100030004FFC01 },
	{ 0x000C04100000FFFF, 0x00100030004FFC01 },
	{ 0x000C04100000FFFF, 0x0000000000000000 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C08140000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x001404080000FFFF, 0x00100030004FFC01 },
	{ 0x001404080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x001000F0004FFC01 },
	{ 0x000C04010000FFFF, 0x001000F0004FFC01 },
	{ 0x000C04010000FFFF, 0x002000F0004FFC01 },
	{ 0x000C04010000FFFF, 0x002000F0004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C04200000FFFF, 0x00100030004FFC01 },
	{ 0x000C04100000FFFF, 0x00100030004FFC01 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C144F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C0C4F0000FFFF, 0x00100030004FFC01 },
	{ 0x000C0C4F0000FFFF, 0x00100030004FFC01 },
	{ 0x001404080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x001424870000FFFF, 0x00100030004FFC01 },
	{ 0x001424870000FFFF, 0x00100030004FFC01 },
	{ 0x000C149E0000FFFF, 0x00100030004FFC01 },
	{ 0x000C149E0000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x00140C050000FFFF, 0x00100030004FFC01 },
	{ 0x0014450E0000FFFF, 0x00100030004FFC01 },
	{ 0x001424870000FFFF, 0x00100030004FFC01 },
	{ 0x0014289E0000FFFF, 0x00000000000FFC00 },
	{ 0x0014289E0000FFFF, 0x00000000000FFC00 },
	{ 0x0014149E0000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x001004080000FFFF, 0x0000000000000000 },
	{ 0x001004080000FFFF, 0x0000000000000000 },
	{ 0x001004080000FFFF, 0x0000000000000000 },
	{ 0x000C00000000FFFF, 0x001000F0004FFC01 },
	{ 0x000C00000000FFFF, 0x001000F0004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x001404080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000C04080000FFFF, 0x00100030004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C04010000FFFF, 0x001001D0004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C04010000FFFF, 0x001001D0004FFC01 },
	{ 0x000000000000FFFF, 0x0000000000000000 },
	{ 0x000C04010000FFFF, 0x001001D0004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x001001D0004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x000C04010000FFFF, 0x00100030004FFC01 },
	{ 0x001404010000FFFF, 0x00100030004FFC01 }
};

static const struct {
	u64	fix;
	u64	be;
} g_qoswt_tbl[] = {
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x000C04050000FFFF, 0x0000000000000000 },
	{ 0x000C080C0000FFFF, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x000C04050000C001, 0x0000000000000000 },
	{ 0x000C080C0000C001, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x001424870000C001, 0x0000000000000000 },
	{ 0x001424870000C001, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x001424870000FFFF, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 },
	{ 0x0000000000000000, 0x0000000000000000 }
};

struct renesas_dbsc5_qos_priv {
	void __iomem		*regs;
};

static int dbsc5_qos_dbsc_setting(struct udevice *dev)
{
	struct renesas_dbsc5_qos_priv *priv = dev_get_priv(dev);
	void __iomem *regs_dbsc_a, *regs_dbsc_d;
	unsigned int ch, nch;

	if (IS_ENABLED(CONFIG_R8A779G0) &&
	    renesas_get_cpu_type() == RENESAS_CPU_TYPE_R8A779G0)
		nch = 2;
	else if (IS_ENABLED(CONFIG_R8A779H0) &&
		 renesas_get_cpu_type() == RENESAS_CPU_TYPE_R8A779H0)
		nch = 1;
	else
		return -EINVAL;

	for (ch = 0; ch < nch; ch++) {
		regs_dbsc_a = priv->regs + DBSC5_DBSC_A_OFFSET + ch * DBSC_A_CH_OFFSET;
		regs_dbsc_d = priv->regs + DBSC5_DBSC_D_OFFSET + ch * DBSC_D_CH_OFFSET;

		/* DBSC CAM, Scheduling Setting */
		writel(0x1234, regs_dbsc_d + DBSC_SYSCNT0);
		writel(0x1234, regs_dbsc_a + DBSC_SYSCNT0);
		writel(0x48218, regs_dbsc_a + DBSC_DBCAM0CNF1);
		writel(0x1C4, regs_dbsc_a + DBSC_DBCAM0CNF2);
		writel(0x3, regs_dbsc_a + DBSC_DBCAM0CNF3);

		if (IS_ENABLED(CONFIG_R8A779G0) &&
		    renesas_get_cpu_type() == RENESAS_CPU_TYPE_R8A779G0 &&
		    (renesas_get_cpu_rev_integer() < 2 ||
		     (renesas_get_cpu_rev_integer() == 2 &&
		      renesas_get_cpu_rev_fraction() <= 1))) {
			/* OTLINT-5579: V4H <= rev2.1 DBSC W/A-3 */
			writel(0x11, regs_dbsc_a + DBSC_DBCAMDIS);
		} else {
			writel(0x10, regs_dbsc_a + DBSC_DBCAMDIS);
		}

		writel(0xF0037, regs_dbsc_a + DBSC_DBSCHCNT0);
		writel(0x1, regs_dbsc_a + DBSC_DBSCHSZ0);
		writel(0xF7311111, regs_dbsc_a + DBSC_DBSCHRW0);
		writel(0x111F1FFF, regs_dbsc_a + DBSC_SCFCTST2);

		/* OTLINT-5579: V4H DBSC WA3 */
		writel(0x7, regs_dbsc_a + DBSC_DBBUS0CNF2);

		/* DBSC QoS Setting */
		writel(0xFFFF, regs_dbsc_a + DBSC_DBSCHQOS_0_0);
		writel(0x480, regs_dbsc_a + DBSC_DBSCHQOS_0_1);
		writel(0x300, regs_dbsc_a + DBSC_DBSCHQOS_0_2);
		writel(0x180, regs_dbsc_a + DBSC_DBSCHQOS_0_3);
		writel(0x400, regs_dbsc_a + DBSC_DBSCHQOS_4_0);
		writel(0x300, regs_dbsc_a + DBSC_DBSCHQOS_4_1);
		writel(0x200, regs_dbsc_a + DBSC_DBSCHQOS_4_2);
		writel(0x100, regs_dbsc_a + DBSC_DBSCHQOS_4_3);
		writel(0x300, regs_dbsc_a + DBSC_DBSCHQOS_9_0);
		writel(0x240, regs_dbsc_a + DBSC_DBSCHQOS_9_1);
		writel(0x180, regs_dbsc_a + DBSC_DBSCHQOS_9_2);
		writel(0xC0, regs_dbsc_a + DBSC_DBSCHQOS_9_3);
		writel(0x40, regs_dbsc_a + DBSC_DBSCHQOS_12_0);
		writel(0x30, regs_dbsc_a + DBSC_DBSCHQOS_12_1);
		writel(0x20, regs_dbsc_a + DBSC_DBSCHQOS_12_2);
		writel(0x10, regs_dbsc_a + DBSC_DBSCHQOS_12_3);
		writel(0x300, regs_dbsc_a + DBSC_DBSCHQOS_13_0);
		writel(0x240, regs_dbsc_a + DBSC_DBSCHQOS_13_1);
		writel(0x180, regs_dbsc_a + DBSC_DBSCHQOS_13_2);
		writel(0xC0, regs_dbsc_a + DBSC_DBSCHQOS_13_3);
		writel(0x200, regs_dbsc_a + DBSC_DBSCHQOS_14_0);
		writel(0x180, regs_dbsc_a + DBSC_DBSCHQOS_14_1);
		writel(0x100, regs_dbsc_a + DBSC_DBSCHQOS_14_2);
		writel(0x80, regs_dbsc_a + DBSC_DBSCHQOS_14_3);
		writel(0x100, regs_dbsc_a + DBSC_DBSCHQOS_15_0);
		writel(0xC0, regs_dbsc_a + DBSC_DBSCHQOS_15_1);
		writel(0x80, regs_dbsc_a + DBSC_DBSCHQOS_15_2);
		writel(0x40, regs_dbsc_a + DBSC_DBSCHQOS_15_3);

		/* Target register is only DBSC0 side. */
		if (ch == 0)
			writel(0x1, regs_dbsc_a + DBSC_FCPRSCTRL);

		writel(0x1, regs_dbsc_a + DBSC_SYSCNT1);
		writel(0x0, regs_dbsc_d + DBSC_SYSCNT0);
		writel(0x0, regs_dbsc_a + DBSC_SYSCNT0);
	}

	return 0;
}

static int dbsc5_qos_settings_init(struct udevice *dev)
{
	struct renesas_dbsc5_qos_priv *priv = dev_get_priv(dev);
	void __iomem *regs_axmm = priv->regs + DBSC5_AXMM_OFFSET;
	void __iomem *regs_cci = priv->regs + DBSC5_CCI_OFFSET;
	void __iomem *regs_qos = priv->regs + DBSC5_QOS_OFFSET;
	int i;

	if (IS_ENABLED(CONFIG_R8A779G0) &&
	    renesas_get_cpu_type() == RENESAS_CPU_TYPE_R8A779G0) {
		/* Address Split 2ch */
		writel(0x0, regs_axmm + AXMM_ADSPLCR0);
		writel(0xFF1B0C, regs_axmm + AXMM_ADSPLCR1);
		writel(0x0, regs_axmm + AXMM_ADSPLCR2);
		writel(0x0, regs_axmm + AXMM_ADSPLCR3);

		writel(0x8000000, regs_cci + CCIQOS00);
		writel(0x8000000, regs_cci + CCIQOS01);

		if (renesas_get_cpu_rev_integer() >= 2) {
			writel(0x1, regs_cci + CCIQOS10);
			writel(0x1, regs_cci + CCIQOS11);
		} else {
			writel(0x0, regs_cci + CCIQOS10);
			writel(0x0, regs_cci + CCIQOS11);
		}

		/* Resource Alloc setting */
		writel(0x48, regs_qos + QOS_RAS);
	} else if (IS_ENABLED(CONFIG_R8A779H0) &&
	    renesas_get_cpu_type() == RENESAS_CPU_TYPE_R8A779H0) {
		/* Resource Alloc setting */
		writel(0x30, regs_qos + QOS_RAS);
	} else {
		return -EINVAL;
	}

	writel(0x2020201, regs_qos + QOS_DANN_LOW);
	writel(0x4040200, regs_qos + QOS_DANN_HIGH);
	writel(0x181008, regs_qos + QOS_DANT);
	writel(0x0, regs_qos + QOS_EMS_LOW);
	writel(0x0, regs_qos + QOS_EMS_HIGH);
	writel(0xA, regs_qos + QOS_FSS);
	writel(0x30F0001, regs_qos + QOS_INSFC);
	writel(0x0, regs_qos + QOS_EARLYR);
	writel(0x50003, regs_qos + QOS_RACNT0);
	writel(0x0, regs_qos + QOS_STATGEN0);

	/* QoS MSTAT setting */
	writel(0x70120, regs_qos + QOS_SL_INIT);
	writel(0x11B0000, regs_qos + QOS_REF_ARS);
	writel(0x12, regs_qos + QOS_REF_ENBL);
	writel(0x4, regs_qos + QOS_BWG);

	if (IS_ENABLED(CONFIG_R8A779G0) &&
	    renesas_get_cpu_type() == RENESAS_CPU_TYPE_R8A779G0 &&
	    (renesas_get_cpu_rev_integer() < 2 ||
	     (renesas_get_cpu_rev_integer() == 2 &&
	      renesas_get_cpu_rev_fraction() <= 1))) {
		/* OTLINT-5579: V4H <= rev2.1 DBSC W/A-3 */
		writel(0x0, regs_axmm + AXMM_MMCR);
	} else {
		writel(0x10000, regs_axmm + AXMM_MMCR);
	}

	writel(0x3, ACTEXT_RT0_R);
	writel(0x3, ACTEXT_RT0_W);

	/*
	 * This may be necessary, but this IP is powered off at this point:
	 * writel(0x3, ACTEXT_IR0_R);
	 * writel(0x3, ACTEXT_IR0_W);
	 * writel(0x3, ACTEXT_IR1_R);
	 * writel(0x3, ACTEXT_IR1_W);
	 */

	if (IS_ENABLED(CONFIG_R8A779G0) &&
	    renesas_get_cpu_type() == RENESAS_CPU_TYPE_R8A779G0) {
		writel(0x10000, regs_axmm + AXMM_TR3CR);

		if (renesas_get_cpu_rev_integer() >= 2) {
			/* WA1 patch for IPL CA76 hang-up issue, REL_TRI_DN-7592 */
			writel(0x38, SI0_RW_MAX);
			writel(0x38, SI1_RW_MAX);
		}
	}

	if (IS_ENABLED(CONFIG_R8A779H0) &&
	    renesas_get_cpu_type() == RENESAS_CPU_TYPE_R8A779H0) {
		writel(0x0, regs_axmm + AXMM_TR0CR0);
		writel(0x0, regs_axmm + AXMM_TR1CR0);
		writel(0x0, regs_axmm + AXMM_TR2CR0);
		writel(0x0, regs_axmm + AXMM_TR3CR0);
		writel(0x70707070, regs_axmm + AXMM_TR0CR1);
		writel(0x70707070, regs_axmm + AXMM_TR1CR1);
		writel(0x70707070, regs_axmm + AXMM_TR2CR1);
		writel(0x70707070, regs_axmm + AXMM_TR3CR1);
		writel(0x70707070, regs_axmm + AXMM_TR0CR2);
		writel(0x70707070, regs_axmm + AXMM_TR1CR2);
		writel(0x70707070, regs_axmm + AXMM_TR2CR2);
		writel(0x70707070, regs_axmm + AXMM_TR3CR2);
	}

	for (i = 0U; i < ARRAY_SIZE(g_qosbw_tbl); i++) {
		writeq(g_qosbw_tbl[i].fix, regs_qos + QOS_FIX_QOS_BANK0 + (i * 8));
		writeq(g_qosbw_tbl[i].fix, regs_qos + QOS_FIX_QOS_BANK1 + (i * 8));
		writeq(g_qosbw_tbl[i].be, regs_qos + QOS_BE_QOS_BANK0 + (i * 8));
		writeq(g_qosbw_tbl[i].be, regs_qos + QOS_BE_QOS_BANK1 + (i * 8));
	}

	for (i = 0U; i < ARRAY_SIZE(g_qoswt_tbl); i++) {
		writeq(g_qoswt_tbl[i].fix, regs_qos + QOSWT_FIX_QOS_BANK0 + (i * 8));
		writeq(g_qoswt_tbl[i].fix, regs_qos + QOSWT_FIX_QOS_BANK1 + (i * 8));
		writeq(g_qoswt_tbl[i].be, regs_qos + QOSWT_BE_QOS_BANK0 + (i * 8));
		writeq(g_qoswt_tbl[i].be, regs_qos + QOSWT_BE_QOS_BANK1 + (i * 8));
	}

	/* QoS SRAM setting */
	writel(0x1, regs_qos + QOS_RAEN);
	writel(0x2080208, regs_qos + QOSWT_WTREF);
	writel(0xD90050F, regs_qos + QOSWT_WTSET0);
	writel(0xD90050F, regs_qos + QOSWT_WTSET1);
	writel(0x1, regs_qos + QOSWT_WTEN);
	writel(0x101, regs_qos + QOS_STATQC);

	return 0;
}

static int renesas_dbsc5_qos_probe(struct udevice *dev)
{
	int ret;

	/* Setting the register of DBSC4 for QoS initialize */
	ret = dbsc5_qos_dbsc_setting(dev);
	if (ret)
		return ret;

	return dbsc5_qos_settings_init(dev);
}

static int renesas_dbsc5_qos_of_to_plat(struct udevice *dev)
{
	struct renesas_dbsc5_qos_priv *priv = dev_get_priv(dev);

	priv->regs = dev_read_addr_ptr(dev);
	if (!priv->regs)
		return -EINVAL;

	return 0;
}

U_BOOT_DRIVER(renesas_dbsc5_qos) = {
	.name		= "dbsc5_qos",
	.id		= UCLASS_NOP,
	.of_to_plat	= renesas_dbsc5_qos_of_to_plat,
	.probe		= renesas_dbsc5_qos_probe,
	.priv_auto	= sizeof(struct renesas_dbsc5_qos_priv),
};
